/*!
 * @file text.h
 * @brief Definition of a text Class, that is able to render text.
*/

#ifndef _TEXT_H
#define _TEXT_H

#include "element_2d.h"

#include "glincl.h"

#define  TEXT_ALIGN_LEFT             E2D_ALIGN_LEFT
#define  TEXT_ALIGN_RIGHT            E2D_ALIGN_RIGHT
#define  TEXT_ALIGN_CENTER           E2D_ALIGN_CENTER
#define  TEXT_ALIGN_SCREEN_CENTER    E2D_ALIGN_SCREEN_CENTER
#define  TEXT_DEFAULT_COLOR          Vector(1.0, 1.0, 1.0)      //!< the default Color (white)
#define  TEXT_DEFAULT_BLENDING       1.0f                       //!< the default blending of the text, (no blending at all)

#define TEXT_DEFAULT_ALIGNMENT       TEXT_ALIGN_CENTER          //!< default alignment
#define TEXT_DEFAULT_SIZE            20                         //!< default size of the Text


// FORWARD DECLARATION
class Font;

/**
 * STATIC means: a font, that is only one GL-face.
 ** it is very fast, and can be used for all text
 ** that does not have to be changed anymore, or if
 ** the the text should look very nice
 * DYNAMIC means: a very fast font, that will is build
 ** from multiple quads.
 ** Use this type, if you want to create fast changing
 ** text like a counter.
 */
typedef enum TEXT_RENDER_TYPE
{
  TEXT_RENDER_STATIC      = 1,
  TEXT_RENDER_DYNAMIC     = 2
};

//! A Struct to handel Texture Coordinates for quads
struct TexCoord
{
  float    minU;              //!< The minimum U-Coordinate
  float    maxU;              //!< The maximum U-Coordinate
  float    minV;              //!< The minimum V-Coordinate
  float    maxV;              //!< The maximum V-Coordinate
};

//! Represents one textElement.
class Text : public Element2D
{
  public:
    Text(const char* fontFile, unsigned int fontSize = TEXT_DEFAULT_SIZE, TEXT_RENDER_TYPE type = TEXT_RENDER_DYNAMIC);
    ~Text();
    void init();

    void setFont(const char* fontFile, unsigned int renderSize);

    void setText(const char* text, bool isExtern = false);

    /** @returns the String this Text displays */
    inline const char* getText() const { return (externText == NULL)?this->text:this->externText; };
    /** @param blending the blending intensity to set (between 0.0 and 1.0) */
    inline void setBlending(float blending) { this->blending = blending; };
    /** sets the Color of the Text to render (values in [0-1]) @param r red @param g green @param b blue */
    void setColor(float r, float g, float b) { this->color = Vector(r, g, b); };
    /** sets the Size of the Font */
    void setSize(float size) { this->size = size; };
    /** @returns the Size of the Text */
//    void getSize(float &x, float& y) const { return this->size; };

    void setType(TEXT_RENDER_TYPE type);
    void createTexture();

    virtual void draw() const;

    void debug() const;

  // helpers.
    static GLuint loadTexture(SDL_Surface* surface, TexCoord* texCoord);
    static int powerOfTwo(int input);

  private:
    Font*             font;           //!< Font of this text

    TEXT_RENDER_TYPE  type;           //!< The type of this Font.
    char*             text;           //!< The text to display
    const char*       externText;     //!< the text to Display from an external Source.
    Vector            color;          //!< The color of the font.
    float             blending;       //!< The blending intensity.
    float             size;           //!< The size of the Font.

  // placement in openGL
    GLuint            texture;        //!< A GL-texture to hold the text (static Mode)
    TexCoord          texCoord;       //!< Texture-coordinates @todo fix this to have a struct
    float             height;
    float             width;
};

#endif /* _TEXT_H */
