/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

   for some fonts and licenses visit: =http://www.dafont.com/en/font.php=

   !! IMPORTANT !! When using ttf fonts clear the license issues prior to
   adding them to orxonox. This is really important, because we do not
   want to offend anyone.
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_FONT

#include "text_engine.h"
#include "text.h"
#include "font.h"

using namespace std;

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "graphics_engine.h"
#include "resource_manager.h"
#include "class_list.h"

#include "debug.h"
#include "list.h"

///////////////////
/// TEXT-ENGINE ///
///////////////////
/**
 *  standard constructor
*/
TextEngine::TextEngine ()
{
   this->setClassID(CL_TEXT_ENGINE, "TextEngine");
   this->setName("TextEngine");
   this->enableFonts();
}

/**
 *  the singleton reference to this class
*/
TextEngine* TextEngine::singletonRef = NULL;

/**
 *  standard deconstructor

*/
TextEngine::~TextEngine ()
{
  // first remove all the remaining Texts (if any).
  tList<BaseObject>* textList = ClassList::getList(CL_TEXT);
  if (textList != NULL)
  {
    tIterator<BaseObject>* textIterator = textList->getIterator();
    Text* text = dynamic_cast<Text*>(textIterator->firstElement());
    while( text != NULL)
    {
      delete text;
      text = dynamic_cast<Text*>(textIterator->nextElement());
    }
    delete textIterator;
  }
  // delete all remaining fonts (this is done in the ResourceManager)
  tList<BaseObject>* fontList = ClassList::getList(CL_FONT);
  if (fontList != NULL)
  {
    tIterator<BaseObject>* fontIterator = fontList->getIterator();
    Font* font = dynamic_cast<Font*>(fontIterator->firstElement());
    while( font != NULL)
    {
      ResourceManager::getInstance()->unload(font, RP_GAME);
      font = dynamic_cast<Font*>(fontIterator->nextElement());
    }
    delete fontIterator;
  }

  this->disableFonts();

  TextEngine::singletonRef = NULL;
}

/**
 *  function to enable TTF_Fonts
*/
void TextEngine::enableFonts()
{
  if (!TTF_WasInit())
    {
      if(TTF_Init()==-1)
        PRINTF(1)("TTF_Init: %s\n", TTF_GetError());

      TextEngine::checkVersion();
    }
  else
    PRINTF(4)("Fonts already initialized\n");
}

/**
 *  function to disable TTF_fonts
*/
void TextEngine::disableFonts()
{
  if (TTF_WasInit())
    {
      Font::removeDefaultFont();
      TTF_Quit();
    }
  else
    PRINTF(4)("Fonts were not initialized.\n");
}

/**
 *  outputs some nice Debug information

   @todo there should also be something outputted about Font
*/
void TextEngine::debug() const
{
  tList<BaseObject>* textList = ClassList::getList(CL_TEXT);
  if (textList != NULL)
  {
    PRINT(0)("+-------------------------------+\n");
    PRINT(0)("+ TEXT ENGINE DEBUG INFORMATION +\n");
    PRINT(0)("+-------------------------------+\n");
    PRINT(0)("Reference: %p; Text Counts: %d\n", this, textList->getSize());

    tIterator<BaseObject>* textIterator = textList->getIterator();
    Text* text = dynamic_cast<Text*>(textIterator->firstElement());
    while( text != NULL)
      {
        text->debug();
        text = dynamic_cast<Text*>(textIterator->nextElement());
      }
    delete textIterator;
    PRINT(0)("+---------------------------TE--+\n");
  }
}


/**
 *  checks if the compiled version and the local version of SDL_ttf match.
 * @returns true if match, false otherwise
*/
bool TextEngine::checkVersion()
{
  SDL_version compile_version;
  SDL_version link_version;
  TTF_VERSION(&compile_version);
  link_version = *TTF_Linked_Version();

  if (compile_version.major == link_version.major &&
      compile_version.minor == link_version.minor &&
      compile_version.patch == link_version.patch)
    {
      return true;
    }
  else
    {
      PRINTF(2)("compiled with SDL_ttf version: %d.%d.%d\n",
                compile_version.major,
                compile_version.minor,
                compile_version.patch);

      PRINTF(2)("running with SDL_ttf version: %d.%d.%d\n",
                link_version.major,
                link_version.minor,
                link_version.patch);
      return false;
    }
}
