/*!
 * @file font.h
 * brief Definition of the FONT-loading class
 *
 * !! IMPORTANT !! When using ttf fonts clear the license issues prior to
 * adding them to orxonox. This is really important, because we do not want
 * to offend anyone.
 */

#ifndef _FONT_H
#define _FONT_H

#include "base_object.h"

#include "glincl.h"


#ifdef HAVE_SDL_TTF_H
#include <SDL_ttf.h>
#else
#include <SDL/SDL_ttf.h>
#endif

/* some default values */
#define FONT_NUM_COLORS              256                        //!< number of colors.

#define FONT_HIGHEST_KNOWN_CHAR      128                        //!< The highest character known to the textEngine.

// FORWARD DECLARATION

//! A struct for handling glyphs
/**
   a Glyph is one letter of a certain font
 */
struct Glyph
{
  // Glyph-specific (size and so on)
  Uint16   character;         //!< The character
  int      minX;              //!< The minimum distance from the origin in X
  int      maxX;              //!< The maximum distance from the origin in X
  int      minY;              //!< The minimum distance from the origin in Y
  int      maxY;              //!< The maximum distance from the origin in Y
  int      width;             //!< The width of the Glyph
  int      height;            //!< The height of the Glyph
  int      bearingX;          //!< How much is right of the Origin
  int      bearingY;          //!< How much is above the Origin
  int      advance;           //!< How big a Glyph would be in monospace-mode

  // OpenGL-specific
  //  TexCoord texCoord;      //!< A Texture Coordinate for this glyph.
  GLuint   displayList;       //!< DiplayList to render this Glyph.
};


////////////
/// FONT ///
////////////
//! A class to handle a Font of a certain ttf-File, Size and Color.
class Font : public BaseObject
{
  friend class Text;

  public:
    Font(const char* fontFile,
         unsigned int fontSize);
    Font(char** xpmArray);
    virtual ~Font();

    void init();

  // font
    bool loadFont(const char* fontFile);
    bool loadFontFromSDL_Surface(SDL_Surface* surface);

    void setSize(unsigned int fontSize);
    void setStyle(const char* renderStyle);

    /** @returns a Pointer to the Array of Glyphs */
    inline Glyph** getGlyphArray() const { return this->glyphArray; };
    /** @returns the texture to the fast-texture */
    inline GLuint getFastTextureID() const { return this->fastTextureID; };
    /** @returns the default Font */
    inline static Font* getDefaultFont() { return Font::defaultFont; };

    void createAsciiImage(const char* fileName);
    static void initDefaultFont();
    static void removeDefaultFont();

  private:
    int getMaxHeight();
    int getMaxAscent();
    int getMaxDescent();
    Glyph* getGlyphMetrics(Uint16 character);

    GLuint createFastTexture();

    void initGlyphs(Uint16 from, Uint16 count);
    int findOptimalFastTextureSize();

    void debug();

  private:
    static Font*  defaultFont;         //!< a default font, that is used, if other fonts were unable to be loaded.
  // information about the Font
    TTF_Font*     font;                //!< The font we use for this.
    unsigned int  fontSize;            //!< The size of the font in pixels. each Font has one size.
    int           renderStyle;         //!< The Renderstyle

    Glyph**       glyphArray;          //!< An Array of all the Glyphs stored in the Array of Glyphs.
    GLuint        fastTextureID;       //!< The fast textureID.
};

#endif /* _FONT_H */
