/*!
 * @file text_engine.h
  *  Definition of textEngine, the Font and the Text

    Text is the text outputed.
    Font is a class that loads a certain ttf-file with a specific height into memory
    TextEngine is used to manage the all the different Fonts that might be included

    for more information see the specific classes.

    !! IMPORTANT !! When using ttf fonts clear the license issues prior to
   adding them to orxonox. This is really important, because we do not want
   to offend anyone.
*/

#ifndef _TEXT_ENGINE_H
#define _TEXT_ENGINE_H


#include "glincl.h"

#ifdef HAVE_SDL_TTF_H
#include <SDL_ttf.h>
#else
#include <SDL/SDL_ttf.h>
#endif

#include "base_object.h"
#include "element_2d.h"

#include "vector.h"

// FORWARD DECLARATION
class PNode;
class Font;

#define  TEXT_ALIGN_LEFT             E2D_ALIGN_LEFT
#define  TEXT_ALIGN_RIGHT            E2D_ALIGN_RIGHT
#define  TEXT_ALIGN_CENTER           E2D_ALIGN_CENTER
#define  TEXT_ALIGN_SCREEN_CENTER    E2D_ALIGN_SCREEN_CENTER
#define  TEXT_DEFAULT_COLOR          Vector(1.0, 1.0, 1.0)      //!< the default Color (white)
#define  TEXT_DEFAULT_BLENDING       1.0f                       //!< the default blending of the text, (no blending at all)

/* some default values */
#define FONT_DEFAULT_SIZE            50                         //!< default size of the Text
#define FONT_NUM_COLORS              256                        //!< number of colors.

#define FONT_HIGHEST_KNOWN_CHAR      128                        //!< The highest character known to the textEngine.

#define TEXT_DEFAULT_ALIGNMENT       TEXT_ALIGN_CENTER          //!< default alignment

typedef enum TEXT_RENDER_TYPE
{
  TEXT_RENDER_STATIC      = 1,
  TEXT_RENDER_DYNAMIC     = 2
};
/**
 * STATIC means: a font, that is only one GL-face.
 ** it is very fast, and can be used for all text
 ** that does not have to be changed anymore, or if
 ** the the text should look very nice
 * DYNAMIC means: a very fast font, that will is build
 ** from multiple quads.
 ** Use this type, if you want to create fast changing
 ** text like a counter.
 */


//! A Struct to handel Texture Coordinates for quads
struct TexCoord
{
  float    minU;              //!< The minimum U-Coordinate
  float    maxU;              //!< The maximum U-Coordinate
  float    minV;              //!< The minimum V-Coordinate
  float    maxV;              //!< The maximum V-Coordinate
};

//! A struct for handling glyphs
/**
   a Glyph is one letter of a certain font
*/
struct Glyph
{
  // Glyph-specific (size and so on)
  Uint16   character;         //!< The character
  int      minX;              //!< The minimum distance from the origin in X
  int      maxX;              //!< The maximum distance from the origin in X
  int      minY;              //!< The minimum distance from the origin in Y
  int      maxY;              //!< The maximum distance from the origin in Y
  int      width;             //!< The width of the Glyph
  int      height;            //!< The height of the Glyph
  int      bearingX;          //!< How much is right of the Origin
  int      bearingY;          //!< How much is above the Origin
  int      advance;           //!< How big a Glyph would be in monospace-mode

  // OpenGL-specific
  //  TexCoord texCoord;      //!< A Texture Coordinate for this glyph.
  GLuint   displayList;       //!< DiplayList to render this Glyph.
};

////////////
/// TEXT ///
////////////
//! Represents one textElement.
class Text : public Element2D
{
  friend class TextEngine;
 public:
   Text(const char* fontFile, unsigned int fontSize = FONT_DEFAULT_SIZE, TEXT_RENDER_TYPE type = TEXT_RENDER_DYNAMIC);
  ~Text();

  void init();

  void setFont(const char* fontFile, unsigned int fontSize);
  void setType(TEXT_RENDER_TYPE type);
  void setText(const char* text, bool isExtern = false);
  /** @returns the String this Text displays */
  inline const char* getText() const { return (externText == NULL)?this->text:this->externText; };
  /** @param blending the blending intensity to set (between 0.0 and 1.0) */
  inline void setBlending(float blending) { this->blending = blending; };

  /** sets the Color of the Text to render (values in [0-1]) @param r red @param g green @param b blue */
  void setColor(float r, float g, float b) { this->color = Vector(r,g,b); };

  void createTexture();

  virtual void draw() const;

  void debug() const;

  // helpers.
  static GLuint loadTexture(SDL_Surface* surface, TexCoord* texCoord);
  static int powerOfTwo(int input);

 private:
   Text(Font* font = NULL, TEXT_RENDER_TYPE type = TEXT_RENDER_DYNAMIC);
   void setFont(Font* font);


 private:
  Font*             font;           //!< Font of this text

  TEXT_RENDER_TYPE  type;           //!< The type of this Font.
  char*             text;           //!< The text to display
  const char*       externText;     //!< the text to Display from an external Source.
  Vector            color;          //!< The color of the font.
  float             blending;       //!< The blending intensity.

  // placement in openGL
  GLuint            texture;        //!< A GL-texture to hold the text
  TexCoord          texCoord;       //!< Texture-coordinates @todo fix this to have a struct
  float             height;
  float             width;
};

////////////
/// FONT ///
////////////
//! A class to handle a Font of a certain ttf-File, Size and Color.
class Font : public BaseObject
{
  friend class Text;

 public:
  Font(const char* fontFile,
       unsigned int fontSize = FONT_DEFAULT_SIZE);
  Font(char** xpmArray);
  virtual ~Font();

  void init();

  // font
  bool loadFont(const char* fontFile);
  bool loadFontFromSDL_Surface(SDL_Surface* surface);

  void setSize(unsigned int fontSize);
  void setStyle(const char* renderStyle);

  /** @returns a Pointer to the Array of Glyphs */
  inline Glyph** getGlyphArray() const { return this->glyphArray; };
  /** @returns the texture to the fast-texture */
  inline GLuint getFastTextureID() const { return this->fastTextureID; };
  /** @returns the default Font */
  inline static Font* getDefaultFont() { return Font::defaultFont; };

  void createAsciiImage(const char* fileName);
  static void initDefaultFont();
  static void removeDefaultFont();

 private:
  int getMaxHeight();
  int getMaxAscent();
  int getMaxDescent();
  Glyph* getGlyphMetrics(Uint16 character);

  GLuint createFastTexture();

  void initGlyphs(Uint16 from, Uint16 count);
  int findOptimalFastTextureSize();

  void debug();

 private:
  static Font*  defaultFont;         //!< a default font, that is used, if other fonts were unable to be loaded.
  // information about the Font
  TTF_Font*     font;                //!< The font we use for this.
  unsigned int  fontSize;            //!< The size of the font in pixels. each Font has one size.
  int           renderStyle;         //!< The Renderstyle

  Glyph**       glyphArray;          //!< An Array of all the Glyphs stored in the Array of Glyphs.
  GLuint        fastTextureID;       //!< The fast textureID.

  tList<Text>*  textList;            //!< A list of texts this Font is mapped to.
};

///////////////////
/// TEXT-ENGINE ///
///////////////////
//! A singleton Class that operates as a Handler for generating and rendering Text in 2D
class TextEngine : public BaseObject
{
 public:
  virtual ~TextEngine();
  /** @returns a Pointer to the only object of this Class */
  inline static TextEngine* getInstance() { if (!singletonRef) singletonRef = new TextEngine();  return singletonRef; };

  Text* createText(const char* fontFile,
                   unsigned int fontSize = FONT_DEFAULT_SIZE,
                   int textType = TEXT_RENDER_DYNAMIC);

  void debug() const;

 private:
  TextEngine();
  static TextEngine* singletonRef;

  // general
  static void enableFonts();
  static void disableFonts();
  static bool checkVersion();
};

#endif /* _TEXT_ENGINE_H */
