/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

   for some fonts and licenses visit: =http://www.dafont.com/en/font.php=

   !! IMPORTANT !! When using ttf fonts clear the license issues prior to
   adding them to orxonox. This is really important, because we do not
   want to offend anyone.
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_FONT

#include "text_engine.h"

using namespace std;

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef HAVE_SDL_IMAGE_H
#include <SDL_image.h>
#else
#include <SDL/SDL_image.h>
#endif
#include "font.xpm"


#include "graphics_engine.h"
#include "resource_manager.h"
#include "class_list.h"

#include "p_node.h"
#include "vector.h"
#include "debug.h"
#include "list.h"

////////////
/// TEXT ///
////////////

/**
 *  creates a new Text Element
 * @param fontFile the Font to render this text in
 * @param type The renderType to display this font in
 */
Text::Text(const char* fontFile, unsigned int fontSize, TEXT_RENDER_TYPE type)
{
  this->init();

  if (fontFile != NULL)
    this->setFont(fontFile, fontSize);
  this->setType(type);
}

/**
 *  creates a new Text Element
 * @param font the Font to render this text in
 * @param type The renderType to display this font in
 *
 * this constructor is private, because the user should initialize
 * a text with the TextEngine.
 */
Text::Text(Font* font, TEXT_RENDER_TYPE type)
{
  this->init();

  this->font = font;
  this->setType(type);
}

/**
 *  deletes a Text out of memory
 *
 * This also ereases the text from the textList of the TextEngine
*/
Text::~Text()
{
  if (this->font != NULL)
    ResourceManager::getInstance()->unload(this->font);

  if (this->text)
    delete[] this->text;
}

void Text::init()
{
  this->setClassID(CL_TEXT, "Text");

  // initialize this Text
  this->font = NULL;
  this->text = NULL;
  this->externText = NULL;
  this->setAlignment(TEXT_DEFAULT_ALIGNMENT);
  this->texture = 0;
  this->blending = TEXT_DEFAULT_BLENDING;
  this->color = TEXT_DEFAULT_COLOR;
  this->setType(TEXT_RENDER_DYNAMIC);

  this->setText(NULL);
}

/**
 * sets the Font of this Text to font
 * @param font the Font (normaly from the ResourceManager) to allocate to this Text
 */
void Text::setFont(Font* font)
{
  if (this->font != NULL)
    ResourceManager::getInstance()->unload(this->font);
  this->font = font;
}

/**
 * sets the Font of this Text to font from fontFile
 * @param fontFile the File to load the Font from.
 * @param fontSize the Size of the Font
 */
void Text::setFont(const char* fontFile, unsigned int fontSize)
{
  Font* tmpFont;
  Text* newText;
  Vector tmpVec;

  tmpFont = (Font*)ResourceManager::getInstance()->load(fontFile, TTF, RP_GAME, &fontSize);
  if (tmpFont == NULL)
  {
    PRINTF(2)("Font %s could not be loaded, probably file not found\n", fontFile);
    this->setFont(NULL);
  }
  else
    this->setFont(tmpFont);
}

/**
 *  sets the Type of this Text
 * @param type the type to set.
*/
void Text::setType(TEXT_RENDER_TYPE type)
{
  if (this->font != NULL && this->font->font)
    this->type = type;
  else
    this->type = TEXT_RENDER_DYNAMIC;
}

/**
 *  Sets a new Text to the font
 * @param text the new text to set
*/
void Text::setText(const char* text, bool isExtern)
{
  if (isExtern)
  {
    this->externText = text;

    if (unlikely(this->text != NULL))
    {
      delete[] this->text;
      this->text = NULL;
    }
  }
  else
  {
    this->externText = NULL;
    if (this->text)
      delete[] this->text;
    if (text != NULL)
    {
      this->text = new char[strlen(text)+1];
      strcpy(this->text, text);
    }
    else
      this->text = NULL;
  }

  // setting up the Text-Width if DYNAMIC
  if (this->type & TEXT_RENDER_DYNAMIC && this->getAlignment() != TEXT_ALIGN_LEFT && this->font != NULL)
    {
      Glyph** glyphArray = this->font->getGlyphArray();

      int width = 0;
      const char* tmpText = this->externText;
      if (this->externText == NULL)
        tmpText = this->text;
      if (tmpText != NULL)
      {
        while (*tmpText != '\0')
        {
          if(glyphArray[*tmpText])
          {
            width += glyphArray[*tmpText]->width;
          }
          tmpText++;
        }
        this->width = width;
      }
    }
}

/**
 *  creates a texture out of the given parameters

   this has to be called every time by the user, to if changes were made.
   this is only for TEXT_STATIC-mode
*/
void Text::createTexture()
{
  SDL_Surface* tmpSurf;
  if (this->texture)
    glDeleteTextures(1, &this->texture);
  if (likely(this->font != NULL))
  {
    SDL_Color theColor = { (int)(this->color.x*255), (int)(this->color.y*255), (int)(this->color.z*255) };
    tmpSurf = TTF_RenderText_Blended(this->font->font,
                                     this->text,
                                     theColor);
  }
  if (tmpSurf)
    this->texture = loadTexture(tmpSurf, &this->texCoord);

  this->width = tmpSurf->w;
  this->height = tmpSurf->h;
  SDL_FreeSurface(tmpSurf);
}

/**
 *  draws the Text
*/
void Text::draw() const
{
  glPushMatrix();
  // transform for alignment.
  if (this->getAlignment() == TEXT_ALIGN_RIGHT)
    glTranslatef(-this->width, 0, 0);
  else if (this->getAlignment() == TEXT_ALIGN_CENTER || this->getAlignment() == TEXT_ALIGN_SCREEN_CENTER)
    glTranslatef(-this->width/2, 0, 0);

  // drawing this Text.
  // setting the Blending effects
  glColor4f(this->color.x, this->color.y, this->color.z, this->blending);
  glEnable(GL_BLEND);
  glEnable(GL_TEXTURE_2D);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);

  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, GL_MODULATE );

  if(likely(type & TEXT_RENDER_DYNAMIC ))
    {
      Glyph** glyphArray;
      if (likely (this->font != NULL))
      {
        glyphArray = this->font->getGlyphArray();
        glBindTexture(GL_TEXTURE_2D, font->getFastTextureID());
      }
      else
      {
        if (unlikely(Font::getDefaultFont() == NULL))
          Font::initDefaultFont();
        glyphArray = Font::getDefaultFont()->getGlyphArray();
        glBindTexture(GL_TEXTURE_2D, Font::getDefaultFont()->getFastTextureID());
      }
        glTranslatef(getAbsCoor2D().x, getAbsCoor2D().y, 0);
//      glRotatef(this->getAbsDir2D(), 0,0,1);
      const char* tmpText = this->externText;
      if (this->externText == NULL)
        tmpText = this->text;
      if (likely(tmpText != NULL))
      {
        while (likely(*tmpText != '\0'))
        {
          if(likely(glyphArray[*tmpText] != NULL))
          {
            glCallList(glyphArray[*tmpText]->displayList);
            glTranslatef(glyphArray[*tmpText]->width, 0, 0);
          }
          tmpText++;
        }
      }
    }
  else //(if type & TEXT_RENDER_STATIC)
    {
      glBindTexture(GL_TEXTURE_2D, this->texture);
      glBegin(GL_QUADS);

      glTexCoord2f(this->texCoord.minU, this->texCoord.minV);
      glVertex2f(this->getAbsCoor2D().x,   this->getAbsCoor2D().y  );

      glTexCoord2f(this->texCoord.maxU, this->texCoord.minV);
      glVertex2f(this->getAbsCoor2D().x + this->width, this->getAbsCoor2D().y  );

      glTexCoord2f(this->texCoord.maxU, this->texCoord.maxV);
      glVertex2f(this->getAbsCoor2D().x + this->width, getAbsCoor2D().y + this->height);

      glTexCoord2f(this->texCoord.minU, this->texCoord.maxV);
      glVertex2f(getAbsCoor2D().x, getAbsCoor2D().y + this->height);

      glEnd();

    }
  glPopMatrix();
}

/**
 *  prints out some nice debug information about this text
*/
void Text::debug() const
{
  if (this->externText == NULL)
    PRINT(0)("=== TEXT: %s ===\n", this->text);
  else
    PRINT(0)("=== TEXT: %s ===\n", this->externText);

  if (this->getBindNode())
    PRINT(0)("is bind to %s; ref=%p\n", this->getBindNode()->getName(), this->getBindNode());
  PRINT(0)("Color: %0.2f %0.2f %0.2f\n", this->color.x, this->color.y, this->color.z);
}


////////////
/// UTIL ///
////////////
/**
 *  Loads a Font from an SDL_surface into a texture.
 * @param surface The surface to make the texture of
 * @param texCoord The texture coordinates of the 4 corners of the texture
 * @returns the ID of the texture
*/
GLuint Text::loadTexture(SDL_Surface *surface, TexCoord* texCoord)
{
  GLuint texture;
  int w, h;
  SDL_Surface *image;
  SDL_Rect area;
  Uint32 saved_flags;
  Uint8  saved_alpha;

  /* Use the surface width and height expanded to powers of 2 */
  w = powerOfTwo(surface->w);
  h = powerOfTwo(surface->h);
  if (texCoord != NULL)
    {
      texCoord->minU = 0.0f;
      texCoord->minV = 0.0f;
      texCoord->maxU = (GLfloat)surface->w / w;
      texCoord->maxV = (GLfloat)surface->h / h;
    }
  image = SDL_CreateRGBSurface(SDL_SWSURFACE,
                               w, h,
                               32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                               0x000000FF,
                               0x0000FF00,
                               0x00FF0000,
                               0xFF000000
#else
                               0xFF000000,
                               0x00FF0000,
                               0x0000FF00,
                               0x000000FF
#endif
                               );
  if ( image == NULL ) {
    return 0;
  }

  /* Save the alpha blending attributes */
  saved_flags = surface->flags&(SDL_SRCALPHA|SDL_RLEACCELOK);
  saved_alpha = surface->format->alpha;
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, 0, 0);
  }

  /* Copy the surface into the GL texture image */
  area.x = 0;
  area.y = 0;
  area.w = surface->w;
  area.h = surface->h;
  SDL_BlitSurface(surface, &area, image, &area);

  /* Restore the alpha blending attributes */
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, saved_flags, saved_alpha);
  }

  /* Create an OpenGL texture for the image */
  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexImage2D(GL_TEXTURE_2D,
               0,
               GL_RGBA,
               w, h,
               0,
               GL_RGBA,
               GL_UNSIGNED_BYTE,
               image->pixels);
  SDL_FreeSurface(image); /* No longer needed the data */

  return texture;
}

/**
 *  Quick utility function for texture creation
 * @param input an integer
 * @returns the next bigger 2^n-integer than input
*/
int Text::powerOfTwo(int input)
{
  int value = 1;

  while ( value < input )
    value <<= 1;
  return value;
}


////////////
/// FONT ///
////////////
/**
 * constructs a Font
 * @param fontFile the File to load the font from
 * @param fontSize the Size of the Font in Pixels
 */
Font::Font(const char* fontFile, unsigned int fontSize)
{
  this->init();

  this->setSize(fontSize);

  if (fontFile != NULL)
    this->loadFont(fontFile);

  this->setStyle("c");//TTF_STYLE_NORMAL);

  this->fastTextureID = this->createFastTexture();


//  this->createAsciiImage("test.bmp");
}

/**
 * constructs a Font
 * @param fontFile the File to load the font from
 * @param fontSize the Size of the Font in Pixels
 */
Font::Font(char** xpmArray)
{
  this->init();

  //  this->setSize(fontSize);
  SDL_Surface* image = NULL;
  if (xpmArray != NULL)
    image = IMG_ReadXPMFromArray(xpmArray);
  if (image != NULL)
  {
    this->loadFontFromSDL_Surface(image);
    SDL_FreeSurface(image);
  }
  else
    PRINTF(1)("loading from surface failed: %s\n", IMG_GetError());

}


/**
 * destructs a font
 * this releases the memory a font uses to be opened.
 * deletes the glLists, and the TTF-handler, if present.
 */
Font::~Font()
{
  // deleting all Glyphs
  if (this->glyphArray != NULL)
    {
      for (int i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
      {
        if (this->glyphArray[i] != NULL)
        {
          glDeleteLists(this->glyphArray[i]->displayList, 1);
          delete this->glyphArray[i];
        }
      }
      delete[] this->glyphArray;
    }

  // erease this font out of the memory.
  if (likely(this->font != NULL))
    TTF_CloseFont(this->font);
}

/**
 * initializes a Font (with default values)
 */
void Font::init()
{
    this->setClassID(CL_FONT, "Font");
  // setting default values.
  this->font = NULL;
  this->glyphArray = NULL;
  this->fastTextureID = 0;
}


/**
 * sets The Font.
 * @param fontFile The file containing the font.
 * @returns true if loaded, false if something went wrong, or if a font was loaded before.
*/
bool Font::loadFont(const char* fontFile)
{
  if (!this->getName())
    {
      this->setName(fontFile);

      this->font = TTF_OpenFont(this->getName(), this->fontSize);
      if(!this->font)
        {
          PRINTF(1)("TTF_OpenFont: %s\n", TTF_GetError());
          return false;
        }
      else
          return true;
    }
  else
    {
      PRINTF(2)("Font already initialized, unable to change it now.\n");
      return false;
    }
}

/**
 * loads a font From an XPM-array.
 * @param xpmArray the array of the XPM to load the font from.
 */
bool Font::loadFontFromSDL_Surface(SDL_Surface* surface)
{
  // loading to a texture.
  if(surface == NULL)
    return false;
  TexCoord texCoord;
  this->fastTextureID = Text::loadTexture(surface, &texCoord);

  // initializing the Glyphs.
  if (this->glyphArray == NULL)
  {
    float cx,cy;
    Glyph* glyph;
    this->glyphArray = new Glyph*[FONT_HIGHEST_KNOWN_CHAR];
    for (int i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
    {
      glyph = this->glyphArray[i] = new Glyph;
      glyph->displayList = glGenLists(1);
      if (!glIsList(glyph->displayList))
      {
        PRINTF(2)("Error creating glList for Font character %c\n", i);
        this->glyphArray[i] = NULL;
        delete glyph;
        continue;
      }
      cx=(float)(i%16)/16.0f;                  // X Position Of Current Character
      cy=(float)(i/16)/16.0f;                  // Y Position Of Current Character
      glNewList(glyph->displayList, GL_COMPILE); // Start Building A List
       glBegin(GL_QUADS);                           // Use A Quad For Each Character
        glTexCoord2f(cx, cy+0.001f);            // Texture Coord (Bottom Left)
        glVertex2d(0,-16);                            // Vertex Coord (Bottom Left)
        glTexCoord2f(cx+0.0625f, cy+0.001f);    // Texture Coord (Bottom Right)
        glVertex2i(16,-16);                           // Vertex Coord (Bottom Right)
        glTexCoord2f(cx+0.0625f, cy+0.0625f);     // Texture Coord (Top Right)
        glVertex2i(16,0);                            // Vertex Coord (Top Right)
        glTexCoord2f(cx, cy+0.0625f);             // Texture Coord (Top Left)
        glVertex2i(0,0);                             // Vertex Coord (Top Left)
       glEnd();                                     // Done Building Our Quad (Character)
//       glTranslated(12,0,0);                        // Move To The Right Of The Character
      glEndList();                                 // Done Building The Display List
      this->glyphArray[i]->width = 12;
    }
  }
  return true;
}


/**
 *  sets a specific renderStyle
 * @param renderStyle the Style to render: a char-array containing:
   i: italic, b: bold, u, underline
*/
void Font::setStyle(const char* renderStyle)
{
  this->renderStyle = TTF_STYLE_NORMAL;

  for (int i = 0; i < strlen(renderStyle); i++)
    if (strncmp(renderStyle+i, "b", 1) == 0)
      this->renderStyle |= TTF_STYLE_BOLD;
    else if (strncmp(renderStyle+i, "i", 1) == 0)
      this->renderStyle |= TTF_STYLE_ITALIC;
    else if (strncmp(renderStyle+i, "u", 1) == 0)
      this->renderStyle |= TTF_STYLE_UNDERLINE;

  if (likely(this->font != NULL))
    TTF_SetFontStyle(this->font, this->renderStyle);
  else
    PRINTF(2)("Font was not initialized, please do so before setting the Font-Style.\n");
}

/**
 *  Sets a new Size to the font
 * @param fontSize The new Size in pixels.
*/
void Font::setSize(unsigned int fontSize)
{
  this->fontSize = fontSize;
}

Font* Font::defaultFont = NULL;

void Font::createAsciiImage(const char* fileName)
{
  if (this->font == NULL)
    return;
  int height = this->getMaxHeight();

  //
  SDL_Color tmpColor = {0, 0, 0};
  // Surface definition.
  SDL_Rect tmpRect; // this represents a Rectangle for blitting.
  SDL_Surface* tmpSurf =  SDL_CreateRGBSurface(SDL_SWSURFACE,
      height*16, height*16,
      32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                                               0x000000FF,
                                               0x0000FF00,
                                               0x00FF0000,
                                               0xFF000000
#else
                                                   0xFF000000,
                                               0x00FF0000,
                                               0x0000FF00,
                                               0x000000FF
#endif
                                                );
    tmpRect.x = 0; tmpRect.y = 0; tmpRect.w = tmpSurf->w; tmpRect.h = tmpSurf->h;
    SDL_SetClipRect(tmpSurf, &tmpRect);
    int maxLineHeight = 0;

    int posX, posY;
  // all the interessting Glyphs
    for (posY = 0; posY < 16; posY++)
    {
      for (posX = 0; posX < 16; posX++)
      {
        SDL_Surface* glyphSurf = NULL;
        if (likely(this->font != NULL))
        {
          SDL_Color white = {255, 255, 255};
          glyphSurf = TTF_RenderGlyph_Blended(this->font, posX+16*posY, white);
        }
        if( glyphSurf != NULL )
        {
          tmpRect.x = height*posX;
          tmpRect.y = height*posY;
          SDL_SetAlpha(glyphSurf, 0, 0);

          SDL_BlitSurface(glyphSurf, NULL, tmpSurf, &tmpRect);
          SDL_FreeSurface(glyphSurf);
              // Outputting Glyphs to BMP-files.
/*
          char outname[512];
          if (i < 10)
          sprintf( outname, "%s-glyph-00%d.bmp", this->getName(), i );
          else if (i <100)
          sprintf( outname, "%s-glyph-0%d.bmp", this->getName(), i );
          else
          sprintf( outname, "%s-glyph-%d.bmp", this->getName(), i );
          SDL_SaveBMP(tmpSurf, outname);*/

        }
      }
    }
    SDL_SaveBMP(tmpSurf, fileName);
    SDL_FreeSurface(tmpSurf);
}

/**
 * initializes the default font
 */
void Font::initDefaultFont()
{
  if (Font::defaultFont == NULL)
    Font::defaultFont = new Font(font_xpm);
}

/**
 * deletes the default font
 */
void Font::removeDefaultFont()
{
  if (Font::defaultFont != NULL)
    delete Font::defaultFont;
  Font::defaultFont = NULL;
}


/**
 * @returns the maximum height of the Font, if the font was initialized, 0 otherwise
*/
int Font::getMaxHeight()
{
  if (likely (this->font != NULL))
    return TTF_FontHeight(this->font);
  else
    return 0;
}

/**
 * @returns the maximum ascent of the Font, if the font was initialized, 0 otherwise

   the ascent is the pixels of the font above the baseline
*/
int Font::getMaxAscent()
{
  if (likely(this->font != NULL))
    return TTF_FontAscent(this->font);
  else
    return 0;
}

/**
 * @returns the maximum descent of the Font, if the font was initialized, 0 otherwise

   the descent is the pixels of the font below the baseline
*/
int Font::getMaxDescent()
{
  if (likely(this->font != NULL))
    return TTF_FontDescent(this->font);
  else
    return 0;
}

/**
 * @param character The character to get info about.
 * @returns a Glyph struct of a character. This Glyph is a pointer,
   and MUST be deleted by the user..

   This only works for horizontal fonts. see
   http://freetype.sourceforge.net/freetype2/docs/tutorial/step2.html
   for more info about vertical Fonts
*/
Glyph* Font::getGlyphMetrics(Uint16 character)
{
  Glyph* rg = new Glyph;
  rg->character = character;
  if (likely (this->font!= NULL))
    TTF_GlyphMetrics(this->font, rg->character,
                     &rg->minX, &rg->maxX,
                     &rg->minY, &rg->maxY,
                     &rg->advance);
  rg->height = rg->maxY - rg->minY;
  rg->width = rg->maxX - rg->minX;
  rg->bearingX = (rg->advance - rg->width) / 2;
  rg->bearingY = rg->maxY;
  return rg;
}

/**
 * creates a Fast-Texture of this Font
 */
GLuint Font::createFastTexture()
{
  /* interesting GLYPHS:
   *  32: space
   *  33-47: Special Characters.
   *  48-57: 0-9
   *  58-63: some more special chars (minor)
   *  65-90: A-Z
   *  97-122: a-z
   */
  int numberOfGlyphs = 91;

  this->initGlyphs(32, numberOfGlyphs);
  this->glyphArray[32]->width = fontSize/2; //!< @todo find out the real size of a Space

  int rectSize = this->findOptimalFastTextureSize();

  // setting default values. (maybe not needed afterwards)
  SDL_Color tmpColor;  tmpColor.r = tmpColor.g = tmpColor.b = 0;
  // Surface definition.
  SDL_Rect tmpRect; // this represents a Rectangle for blitting.
  SDL_Surface* tmpSurf =  SDL_CreateRGBSurface(SDL_SWSURFACE,
                                               rectSize, rectSize,
                                               32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                                               0x000000FF,
                                               0x0000FF00,
                                               0x00FF0000,
                                               0xFF000000
#else
                                               0xFF000000,
                                               0x00FF0000,
                                               0x0000FF00,
                                               0x000000FF
#endif
                                               );
  tmpRect.x = 0; tmpRect.y = 0; tmpRect.w = tmpSurf->w; tmpRect.h = tmpSurf->h;
  SDL_SetClipRect(tmpSurf, &tmpRect);
  int maxLineHeight = 0;

  // all the interessting Glyphs
  for (int i = 0; i < 128; i++)
    {
      SDL_Surface* glyphSurf = NULL;
      Glyph* tmpGlyph;

      if (tmpGlyph = this->glyphArray[i])
        {
          if (tmpGlyph->height > maxLineHeight)
            maxLineHeight = tmpGlyph->height;

          if (tmpRect.x+tmpGlyph->advance > tmpSurf->w)
            {
              tmpRect.x = 0;
              tmpRect.y = tmpRect.y + maxLineHeight + 1;
              maxLineHeight = 0;
            }
          if (tmpRect.y + maxLineHeight > tmpSurf->h)
            {
              PRINTF(1)("Protection, so font cannot write over the boundraries error (this should not heappen\n");
              break;
            }
          // reading in the new Glyph
          if (likely(this->font != NULL))
          {
            SDL_Color white = {255, 255, 255};
            glyphSurf = TTF_RenderGlyph_Blended(this->font, i, white);
          }
          if( glyphSurf != NULL )
            {
              SDL_SetAlpha(glyphSurf, 0, 0);

              SDL_BlitSurface(glyphSurf, NULL, tmpSurf, &tmpRect);
              TexCoord tmpTexCoord;
              tmpTexCoord.minU = (float)tmpRect.x/(float)tmpSurf->w;
              tmpTexCoord.maxU = (float)(tmpRect.x + tmpGlyph->advance)/(float)tmpSurf->w;
              tmpTexCoord.minV = (float)(tmpRect.y)/(float)tmpSurf->w;
              tmpTexCoord.maxV = (float)(tmpRect.y+tmpGlyph->height)/(float)tmpSurf->w;
              tmpGlyph->displayList = glGenLists(1);

              glNewList(tmpGlyph->displayList, GL_COMPILE);
              glBegin(GL_QUADS);
              glTexCoord2f(tmpTexCoord.minU, tmpTexCoord.minV);
              glVertex2d(0, - tmpGlyph->bearingY);
              glTexCoord2f(tmpTexCoord.minU, tmpTexCoord.maxV);
              glVertex2d(0, tmpGlyph->height - tmpGlyph->bearingY);
              glTexCoord2f(tmpTexCoord.maxU, tmpTexCoord.maxV);
              glVertex2d(tmpGlyph->width, tmpGlyph->height - tmpGlyph->bearingY);
              glTexCoord2f(tmpTexCoord.maxU, tmpTexCoord.minV);
              glVertex2d(tmpGlyph->width, - tmpGlyph->bearingY);
              glEnd();
              glEndList();
              SDL_FreeSurface(glyphSurf);

              tmpRect.x += tmpGlyph->advance;

              // Outputting Glyphs to BMP-files.
/*
                char outname[512];
                if (i < 10)
                sprintf( outname, "%s-glyph-00%d.bmp", this->getName(), i );
                else if (i <100)
                  sprintf( outname, "%s-glyph-0%d.bmp", this->getName(), i );
                else
                  sprintf( outname, "%s-glyph-%d.bmp", this->getName(), i );
                SDL_SaveBMP(tmpSurf, outname);*/

            }
        }
    }

  GLuint texture;
  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexImage2D(GL_TEXTURE_2D,
               0,
               GL_RGBA,
               tmpSurf->w, tmpSurf->h,
               0,
               GL_RGBA,
               GL_UNSIGNED_BYTE,
               tmpSurf->pixels);
  SDL_FreeSurface(tmpSurf);
  return texture;
}

/**
 *  stores Glyph Metrics in an Array.
 * @param from The Glyph to start from.
 * @param count The number of Glyphs to start From.
*/
void Font::initGlyphs(Uint16 from, Uint16 count)
{
  /* initialize the Array, and set all its entries to NULL
   *  only if the Glyph-array has not been initialized
   */
  if (!this->glyphArray)
    {
      this->glyphArray = new Glyph*[FONT_HIGHEST_KNOWN_CHAR];
      for (int i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
        this->glyphArray[i] = NULL;
    }

  Uint16 lastGlyph = from + count;

  for (int i = from; i <= lastGlyph; i++)
    {
      // setting up all the Glyphs we like.
      glyphArray[i] = getGlyphMetrics(i);
    }
  return;
}

/**
 * @returns the optimal size to use as the texture size

   @todo: this algorithm can be a lot more faster, althought it does
   not really matter within the init-context, and 128 glyphs.

   This function searches for a 2^n sizes texture-size, this is for
   openGL-version < 1.2 compatibility ( and because it is realy easy like this :))
*/
int Font::findOptimalFastTextureSize()
{
  int i;
  int x,y; // the counters
  int maxLineHeight = this->getMaxHeight();
  unsigned int size = 32;  // starting Value, we have to start somewhere 32 seems reasonable. (take any small enough 2^i number)
  bool sizeOK = false;
  Glyph* tmpGlyph;

  while (!sizeOK)
    {
      x = 0; y = 0;
      for (i = 0; i <= FONT_HIGHEST_KNOWN_CHAR; i++)
        {
          if((tmpGlyph = this->glyphArray[i]) != NULL)
            {
              // getting the height of the highest Glyph in the Line.
              if (tmpGlyph->height > maxLineHeight)
                maxLineHeight = tmpGlyph->height;

              if (x + tmpGlyph->advance > size)
                {
                  x = 0;
                  y = y + maxLineHeight;
                  //maxLineHeight = 0;
                }
              if (y + maxLineHeight + 1 > size)
                break;
              x += tmpGlyph->advance;

            }
        }
      if (i >= FONT_HIGHEST_KNOWN_CHAR-1 || size > 8192)
        sizeOK = true;
      else
        size *= 2;
    }
    return size;
}


/**
 *  a simple function to get some interesting information about this class
*/
void Font::debug()
{
  // print the loaded font's style
  int style;
  if (likely(this->font != NULL))
    style = TTF_GetFontStyle(this->font);
  PRINTF(0)("The font style is:");
  if(style==TTF_STYLE_NORMAL)
    PRINTF(0)(" normal");
  else {
    if(style&TTF_STYLE_BOLD)
      PRINTF(0)(" bold");
    if(style&TTF_STYLE_ITALIC)
      PRINTF(0)(" italic");
    if(style&TTF_STYLE_UNDERLINE)
      PRINTF(0)(" underline");
  }
  PRINTF(0)("\n");
}


///////////////////
/// TEXT-ENGINE ///
///////////////////
/**
 *  standard constructor
*/
TextEngine::TextEngine ()
{
   this->setClassID(CL_TEXT_ENGINE, "TextEngine");
   this->setName("TextEngine");
   this->enableFonts();
}

/**
 *  the singleton reference to this class
*/
TextEngine* TextEngine::singletonRef = NULL;

/**
 *  standard deconstructor

*/
TextEngine::~TextEngine ()
{
  // first remove all the remaining Texts (if any).
  tList<BaseObject>* textList = ClassList::getList(CL_TEXT);
  if (textList != NULL)
  {
    tIterator<BaseObject>* textIterator = textList->getIterator();
    Text* text = dynamic_cast<Text*>(textIterator->firstElement());
    while( text != NULL)
    {
      delete text;
      text = dynamic_cast<Text*>(textIterator->nextElement());
    }
    delete textIterator;
  }
  // delete all remaining fonts (this is done in the ResourceManager)
  tList<BaseObject>* fontList = ClassList::getList(CL_FONT);
  if (fontList != NULL)
  {
    tIterator<BaseObject>* fontIterator = fontList->getIterator();
    Font* font = dynamic_cast<Font*>(fontIterator->firstElement());
    while( font != NULL)
    {
      ResourceManager::getInstance()->unload(font, RP_GAME);
      font = dynamic_cast<Font*>(fontIterator->nextElement());
    }
    delete fontIterator;
  }

  this->disableFonts();

  TextEngine::singletonRef = NULL;
}

/**
 *  function to enable TTF_Fonts
*/
void TextEngine::enableFonts()
{
  if (!TTF_WasInit())
    {
      if(TTF_Init()==-1)
        PRINTF(1)("TTF_Init: %s\n", TTF_GetError());

      TextEngine::checkVersion();
    }
  else
    PRINTF(4)("Fonts already initialized\n");
}

/**
 *  function to disable TTF_fonts
*/
void TextEngine::disableFonts()
{
  if (TTF_WasInit())
    {
      Font::removeDefaultFont();
      TTF_Quit();
    }
  else
    PRINTF(4)("Fonts were not initialized.\n");
}

/**
 *  creates a new Text with a certain font.
 * @see Font::Font
 * @see Text::Text
*/
Text* TextEngine::createText(const char* fontFile, unsigned int fontSize, int textType)
{
  Font* tmpFont;
  Text* newText;
  Vector tmpVec;

  tmpFont = (Font*)ResourceManager::getInstance()->load(fontFile, TTF, RP_GAME, &fontSize);
  if (!tmpFont)
    {
      PRINTF(2)("Font %s could not be loaded, probably file not found\n", fontFile);
      return NULL;
    }
  else
    return new Text(tmpFont, TEXT_RENDER_DYNAMIC);
}

/**
 *  outputs some nice Debug information

   @todo there should also be something outputted about Font
*/
void TextEngine::debug() const
{
  tList<BaseObject>* textList = ClassList::getList(CL_TEXT);
  if (textList != NULL)
  {
    PRINT(0)("+-------------------------------+\n");
    PRINT(0)("+ TEXT ENGINE DEBUG INFORMATION +\n");
    PRINT(0)("+-------------------------------+\n");
    PRINT(0)("Reference: %p; Text Counts: %d\n", this, textList->getSize());

    tIterator<BaseObject>* textIterator = textList->getIterator();
    Text* text = dynamic_cast<Text*>(textIterator->firstElement());
    while( text != NULL)
      {
        text->debug();
        text = dynamic_cast<Text*>(textIterator->nextElement());
      }
    delete textIterator;
    PRINT(0)("+---------------------------TE--+\n");
  }
}


/**
 *  checks if the compiled version and the local version of SDL_ttf match.
 * @returns true if match, false otherwise
*/
bool TextEngine::checkVersion()
{
  SDL_version compile_version;
  SDL_version link_version;
  TTF_VERSION(&compile_version);
  link_version = *TTF_Linked_Version();

  if (compile_version.major == link_version.major &&
      compile_version.minor == link_version.minor &&
      compile_version.patch == link_version.patch)
    {
      return true;
    }
  else
    {
      PRINTF(2)("compiled with SDL_ttf version: %d.%d.%d\n",
                compile_version.major,
                compile_version.minor,
                compile_version.patch);

      PRINTF(2)("running with SDL_ttf version: %d.%d.%d\n",
                link_version.major,
                link_version.minor,
                link_version.patch);
      return false;
    }
}
