/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "texture.h"

#include "debug.h"
#include "graphics_engine.h"

#ifdef HAVE_SDL_IMAGE_H
#include <SDL_image.h>
#else
#include <SDL/SDL_image.h>
#endif

/**
 *  Constructor for a Texture
*/
Texture::Texture(const char* imageName)
{
  this->setClassID(CL_TEXTURE, "Texture");
  this->setName(imageName);

  this->bAlpha = false;
  this->texture = 0;
  if (imageName != NULL)
    this->loadImage(imageName);
}

/**
 *  Destructor of a Texture

   Frees Data, and deletes the textures from GL
*/
Texture::~Texture()
{
  if (this->texture != 0)
    glDeleteTextures(1, &this->texture);
}

/**
 *  Loads a Texture to the openGL-environment.
 * @param surface the Image to load to openGL
 * @returns The ID of the texture.
*/
GLuint Texture::loadTexToGL (SDL_Surface* surface)
{
  if (GraphicsEngine::texturesEnabled)
    {
      PRINTF(4)("Loading texture to OpenGL-Environment.\n");

      int w, h;
      SDL_Surface *image;
      SDL_Rect area;
      Uint32 saved_flags;
      Uint8  saved_alpha;

      w = surface->w;
      h = surface->h;

      image = SDL_CreateRGBSurface(SDL_SWSURFACE,
                                   w, h,
                                   32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                                   0x000000FF,
                                   0x0000FF00,
                                   0x00FF0000,
                                   0xFF000000
#else
                                   0xFF000000,
                                   0x00FF0000,
                                   0x0000FF00,
                                   0x000000FF
#endif
                                   );
      if ( image == NULL ) {
        return 0;
      }

      /* Save the alpha blending attributes */
      saved_flags = surface->flags&(SDL_SRCALPHA|SDL_RLEACCELOK);
      saved_alpha = surface->format->alpha;
      if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
        SDL_SetAlpha(surface, 0, 0);
      }

      /* Copy the surface into the GL texture image */
      area.x = 0;
      area.y = 0;
      area.w = surface->w;
      area.h = surface->h;
      SDL_BlitSurface(surface, &area, image, &area);

      /* Restore the alpha blending attributes */
      if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
        SDL_SetAlpha(surface, saved_flags | SDL_OPENGL, saved_alpha);
        this->bAlpha = true;
      }

      /* Create an OpenGL texture for the image */
      glGenTextures(1, &this->texture);
      glBindTexture(GL_TEXTURE_2D, this->texture);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      // build the Texture
      glTexImage2D(GL_TEXTURE_2D,
                   0,
                   GL_RGBA,
                   w, h,
                   0,
                   GL_RGBA,
                   GL_UNSIGNED_BYTE,
                   image->pixels);
      // build the MipMaps
      gluBuild2DMipmaps(GL_TEXTURE_2D,
                        GL_RGBA,
                        w,
                        h,
                        GL_RGBA,
                        GL_UNSIGNED_BYTE,
                        image->pixels);

      SDL_FreeSurface(image); /* No longer needed */
      glBindTexture(GL_TEXTURE_2D, 0);
    }
}

/**
 *  loads an Image from a file to a Texture
 * @param imageName The image to load
*/
bool Texture::loadImage(const char* imageName)
{
  if (GraphicsEngine::texturesEnabled)
    {
      if (imageName != NULL)
        {
          SDL_Surface* tmpSurf;
          if (this->texture)
            glDeleteTextures(1, &this->texture);
          // load the new Image to memory
          tmpSurf = IMG_Load(imageName);
          if(tmpSurf == NULL)
            {
              PRINTF(1)("IMG_Load: %s\n", IMG_GetError());
              this->texture = 0;
              return false;
            }
          else
          {
            PRINTF(3)("loading Image %s\n", imageName);
            loadTexToGL(tmpSurf);
            SDL_FreeSurface(tmpSurf);
            return true;
          }
        }
      else
        {
          return false;
        }
    }
    return false;
}
