/*!
 * @file graphics_engine.h

  *  defines a Interface between orxonox and graphical input

    handles graphical SDL-initialisation, textures, resolutions, and so on
 */

#ifndef _GRAPHICS_ENGINE_H
#define _GRAPHICS_ENGINE_H

#include "event_listener.h"

#include "sdlincl.h"
#include "glincl.h"

// Forward Declaration
class Text;
class IniParser;

//! class to handle graphics
/**
   handles graphical SDL-initialisation, textures, resolutions, and so on
 */
class GraphicsEngine : public EventListener
{
  public:
    virtual ~GraphicsEngine();
    /** @returns a Pointer to the only object of this Class */
    inline static GraphicsEngine* getInstance() { if (!GraphicsEngine::singletonRef) GraphicsEngine::singletonRef = new GraphicsEngine();  return GraphicsEngine::singletonRef; };

    int init();
    int initFromIniFile(IniParser* iniParser);

    void setWindowName(const char* windowName, const char* icon);

    int setResolution(int width, int height, int bpp);
    void setFullscreen(bool fullscreen = false);
    static void setBackgroundColor(GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha = 1.0);


    /** @returns the x resolution */
    inline int getResolutionX() const { return this->resolutionX; };
    /** @returns the y resolution */
    inline int getResolutionY() const { return this->resolutionY; };
    /** @returns the Bits Per Pixel */
    inline int getbbp() const { return this->bitsPerPixel; };

    int resolutionChanged(const SDL_ResizeEvent& resizeInfo);

    static void showMouse(bool show);
    static bool isMouseVisible();
    static void stealWMEvents(bool steal);
    static bool isStealingEvents();

    static void enter2DMode();
    static void leave2DMode();

    static void storeMatrices();
    static GLdouble modMat[16];
    static GLdouble projMat[16];
    static GLint viewPort[4];

    void update(float dt);
    void tick(float dt);
    void draw() const;
    void displayFPS(bool display);

    void listModes();

    /** \brief swaps the GL_BUFFERS */
    inline static void swapBuffers() { SDL_GL_SwapBuffers(); };

    void process(const Event  &event);

  private:
    GraphicsEngine();
    int initVideo(unsigned int resX, unsigned int resY, unsigned int bbp);
    int setGLattribs();

  public:
    static bool             texturesEnabled; //!< if textures should be enabled (globally)


  private:
    static GraphicsEngine*  singletonRef;    //!< Pointer to the only instance of this Class
    bool                    isInit;          //!< if the GraphicsEngine is initialized.

    SDL_Surface*            screen;          //!< the screen we draw to
    int                     resolutionX;     //!< the X-resoultion of the screen
    int                     resolutionY;     //!< the Y-resolution of the screen
    int                     bitsPerPixel;    //!< the bits per pixels of the screen
    Uint32                  fullscreenFlag;  //!< if we are in fullscreen mode
    Uint32                  videoFlags;      //!< flags for video
    SDL_Rect**              videoModes;      //!< list of resolutions

    bool                    bDisplayFPS;     //!< is true if the fps should be displayed
    float                   currentFPS;      //!< the current frame rate: frames per seconds
    float                   maxFPS;          //!< maximal frame rate we ever got since start of the game
    float                   minFPS;          //!< minimal frame rate we ever got since start


#ifndef NO_TEXT
  Text*          geTextCFPS;      //!< Text for the current FPS
  Text*          geTextMaxFPS;    //!< Text for the max FPS
  Text*          geTextMinFPS;    //!< Text for the min FPS
#endif /* NO_TEXT */
};

#endif /* _GRAPHICS_ENGINE_H */
