/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell.h"
#include "shell_command.h"
#include "shell_buffer.h"
#include "shell_input.h"


#include "text_engine.h"
#include "list.h"
#include "graphics_engine.h"
#include "event_handler.h"
#include "debug.h"
#include "class_list.h"

#include "key_names.h"
#include <stdarg.h>
#include <stdio.h>

using namespace std;

SHELL_COMMAND(clear, Shell, clear)
    ->describe("Clears the shell from unwanted lines (empties all buffers)")
    ->setAlias("clear");
SHELL_COMMAND(deactivate, Shell, deactivate)
    ->describe("Deactivates the Shell. (moves it into background)")
    ->setAlias("hide");
SHELL_COMMAND(textsize, Shell, setTextSize)
    ->describe("Sets the size of the Text (size, linespacing)");

/**
 * standard constructor
 */
Shell::Shell ()
{
  this->setClassID(CL_SHELL, "Shell");
  this->setName("Shell");

  // register the shell at the ShellBuffer
  ShellBuffer::getInstance()->registerShell(this);


  // Element2D and generals
  this->setAbsCoor2D(3, -400);
  this->textSize = 15;
  this->lineSpacing = 0;
  this->bActive = false;

  // BUFFER
  this->bufferText = NULL;
  this->setBufferDisplaySize(10);

  // INPUT LINE
  this->shellInput = new ShellInput;
  //this->commandList = new tList<ShellCommand>;

  this->rebuildText();

  // EVENT-Handler subscription of '`' to all States.
  EventHandler::getInstance()->subscribe(this, ES_ALL, SDLK_BACKQUOTE);
}

/**
 * standard deconstructor
 */
Shell::~Shell ()
{
  // delete the displayable Buffers
  for (int i = 0; i < this->bufferDisplaySize; i++)
    delete this->bufferText[i];
  delete[] this->bufferText;

  // delete the inputLine
  delete this->shellInput;
  ShellBuffer::getInstance()->unregisterShell(this);
}

/**
 * activates the shell
 *
 * This also feeds the Last few lines from the main buffers into the displayBuffer
 */
void Shell::activate()
{
  if (this->bActive == true)
    PRINTF(3)("The shell is already active\n");
  this->bActive = true;

  EventHandler::getInstance()->setState(ES_SHELL);
  this->setRelCoorSoft2D(0, 0, 1, 5);

  ShellBuffer::getInstance()->getBufferIterator()->lastElement();
  for (int i = 0; i < this->bufferDisplaySize; i++)
    this->bufferText[i]->setText(ShellBuffer::getInstance()->getBufferIterator()->prevElement(), true);
}

/**
 * deactiveates the Shell.
 */
void Shell::deactivate()
{
  if (this->bActive == false)
    PRINTF(3)("The shell is already inactive\n");
  this->bActive = false;

  EventHandler::getInstance()->setState(ES_GAME);
  this->setRelCoorSoft2D(0, -400, 1, 5);

  ShellBuffer::getInstance()->getBufferIterator()->lastElement();
  for (int i = 0; i < this->bufferDisplaySize; i++)
    this->bufferText[i]->setText(ShellBuffer::getInstance()->getBufferIterator()->prevElement(), false);
}


/**
 * sets the size of the text and spacing
 * @param textSize the size of the Text in Pixels
 * @param lineSpacing the size of the Spacing between two lines in pixels
 *
 * this also rebuilds the entire Text, inputLine and displayBuffer,
 * to be accurate again.
 */
void Shell::setTextSize(unsigned int textSize, unsigned int lineSpacing)
{
  this->textSize = textSize;
  this->lineSpacing = lineSpacing;
  this->shellInput->setFont("fonts/dpquake_.ttf", this->textSize);

//  this->rebuildText();
}

/**
 * rebuilds the Text's
 *
 * use this function, if you changed the Font/Size or something else.
 */
void Shell::rebuildText()
{
  this->shellInput->setFont("fonts/dpquake_.ttf", this->textSize);
  this->shellInput->setColor(1, 0, 0);
  this->shellInput->setAlignment(TEXT_ALIGN_LEFT);
  this->shellInput->setParent2D(this);
  this->shellInput->setRelCoor2D(5, (this->textSize + this->lineSpacing)*this->bufferDisplaySize + this->textSize);

  this->setBufferDisplaySize(this->bufferDisplaySize);
}

/**
 * sets The count of Lines to display in the buffer.
 * @param bufferDisplaySize the count of lines to display in the Shell-Buffer.
 */
void Shell::setBufferDisplaySize(unsigned int bufferDisplaySize)
{
  if (this->bufferText != NULL)
  {
    for (unsigned int i = 0; i < this->bufferDisplaySize; i++)
      delete this->bufferText[i];
    delete[] this->bufferText;
  }

  this->bufferText = new Text*[bufferDisplaySize];
  for (unsigned int i = 0; i < bufferDisplaySize; i++)
  {
    this->bufferText[i] = TextEngine::getInstance()->createText("fonts/dpquake_.ttf", this->textSize, TEXT_RENDER_DYNAMIC);
    this->bufferText[i]->setColor(1, 0, 0);
    this->bufferText[i]->setAlignment(TEXT_ALIGN_LEFT);
    this->bufferText[i]->setRelCoor2D(calculateLinePosition(i));
    this->bufferText[i]->setText(NULL);
    this->bufferText[i]->setParent2D(this);
  }
  this->bufferDisplaySize = bufferDisplaySize;

  this->shellHeight = (this->textSize + this->lineSpacing) * (bufferDisplaySize+1);
}

/**
 * deletes all the Buffers
 */
void Shell::flush()
{
  // remove all chars from the BufferTexts.
  if (this->bufferText)
    for (int i = 0; i < this->bufferDisplaySize; i++)
    {
      this->bufferText[i]->setText(NULL, true);
    }
  // BUFFER FLUSHING
}

/**
 * prints out some text to the input-buffers
 * @param text the text to output.
 */
void Shell::printToDisplayBuffer(const char* text)
{
  if(likely(bufferText != NULL))
  {
    Text* lastText = this->bufferText[this->bufferDisplaySize-1];

    Text* swapText;
    Text* moveText = this->bufferText[0];
    this->bufferText[0]->setRelCoorSoft2D(this->calculateLinePosition(1),10);
    for (unsigned int i = 1; i < this->bufferDisplaySize; i++)
    {
      if ( i < this->bufferDisplaySize-1)
        this->bufferText[i]->setRelCoorSoft2D(this->calculateLinePosition(i+1),5);
      swapText = this->bufferText[i];
      this  ->bufferText[i] = moveText;
      moveText = swapText;
    }
    lastText->setRelCoor2D(this->calculateLinePosition(0));
    this->bufferText[0] = lastText;

    this->bufferText[0]->setText(text, true);
  }
}

/**
 * clears the Shell (empties all buffers)
 */
void Shell::clear()
{
  this->flush();
  ShellBuffer::addBufferLineStatic("orxonox - shell\n ==================== \n", NULL);
}

/**
 * listens for some event
 * @param event the Event happened
 */
void Shell::process(const Event &event)
{
  if (event.bPressed)
  {
    if (event.type == SDLK_BACKQUOTE)
    {
      if (EventHandler::getInstance()->getState() == ES_GAME)
        this->activate();
      else
        this->deactivate();
    }
  }
}

/**
 * displays the Shell
 */
void Shell::draw() const
{
  glPushMatrix();
  // transform for alignment.
  // setting the Blending effects

  glColor4f(0.0f, 0.0f, 0.8f, .4);
  glEnable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);

  glBindTexture(GL_TEXTURE_2D, 0);
  glBegin(GL_TRIANGLE_STRIP);

  glTexCoord2f(0, 0);
  glVertex2f(this->getAbsCoor2D().x,   this->getAbsCoor2D().y  );

  glTexCoord2f(1, 0);
  glVertex2f(GraphicsEngine::getInstance()->getResolutionX() - this->getAbsCoor2D().x, this->getAbsCoor2D().y  );

  glTexCoord2f(0, 1);
  glVertex2f(this->getAbsCoor2D().x, this->getAbsCoor2D().y + this->shellHeight);

  glTexCoord2f(1, 1);
  glVertex2f(GraphicsEngine::getInstance()->getResolutionX() - this->getAbsCoor2D().x, this->getAbsCoor2D().y + this->shellHeight);

  glEnd();
}

///////////////////////
// HELPER FUNCTIONS  //
///////////////////////

/**
 * calculates the position of a Buffer-Display Line
 * @param lineNumber the lineNumber from the bottom to calculate the position from
 * @returns the Position of the Line.
 */
Vector Shell::calculateLinePosition(unsigned int lineNumber)
{
  return Vector(5, (this->textSize + this->lineSpacing)*(this->bufferDisplaySize - lineNumber -1) + this->textSize, 0);
}



/**
 * displays some nice output from the Shell
 */
void Shell::debug() const
{
  PRINT(3)("Debugging output to console (not this shell)\n");

//   if (this->pressedKey != SDLK_FIRST)
//     printf("%s::%f %f\n", SDLKToKeyname(this->pressedKey), this->delayed, this->repeatDelay);


  ShellBuffer::getInstance()->debug();
}

// void Shell::testI (int i)
// {
//   PRINTF(3)("This is the Test for one Int '%d'\n", i);
// }
//
// void Shell::testS (const char* s)
// {
//   PRINTF(3)("This is the Test for one String '%s'\n", s);
// }
//
// void Shell::testB (bool b)
// {
//   PRINTF(3)("This is the Test for one Bool: ");
//   if (b)
//     PRINTF(3)("true\n");
//   else
//     PRINTF(3)("false\n");
// }
//
// void Shell::testF (float f)
// {
//   PRINTF(3)("This is the Test for one Float '%f'\n", f);
// }
//
// void Shell::testSF (const char* s, float f)
// {
//   PRINTF(3)("This is the Test for one String '%s' and one Float '%f'\n",s , f);
// }
