/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell_command.h"

#include "list.h"
#include "debug.h"
#include "class_list.h"

#include "key_names.h"
#include <stdarg.h>
#include <stdio.h>

using namespace std;

/**
 * creates a new ShellCommandClass
 * @param className the Name of the command-class to create
 */
ShellCommandClass::ShellCommandClass(const char* className)
{
  this->className = className;
  this->classID = CL_NULL;
  this->commandList = new tList<ShellCommandBase>;

  ShellCommandClass::commandClassList->add(this);
}

/**
 * destructs the shellCommandClass again
 */
ShellCommandClass::~ShellCommandClass()
{
  tIterator<ShellCommandBase>* iterator = this->commandList->getIterator();
  ShellCommandBase* elem = iterator->firstElement();
  while(elem != NULL)
  {
    delete elem;
    elem = iterator->nextElement();
  }
  delete iterator;
  delete this->commandList;
}

/**
 * unregisters all Commands that exist
 */
void ShellCommandClass::unregisterAllCommands()
{
  tIterator<ShellCommandClass>* iterator = ShellCommandClass::commandClassList->getIterator();
  ShellCommandClass* elem = iterator->firstElement();
  while(elem != NULL)
  {
    delete elem;

    elem = iterator->nextElement();
  }
  delete iterator;

  delete ShellCommandClass::commandClassList;
  ShellCommandClass::commandClassList = NULL;
}

const ShellCommandClass* ShellCommandClass::isRegistered(const char* className)
{
  if (ShellCommandClass::commandClassList == NULL)
    initCommandClassList();

  tIterator<ShellCommandClass>* iterator = ShellCommandClass::commandClassList->getIterator();
  ShellCommandClass* elem = iterator->firstElement();
  while(elem != NULL)
  {
    if (!strcmp(className, elem->className))
    {
      if (elem->classID == CL_NULL)
        elem->classID = ClassList::StringToID(className);

      delete iterator;
      return elem;
    }
    elem = iterator->nextElement();
  }
  delete iterator;
  return NULL;
}

ShellCommandClass* ShellCommandClass::getCommandClass(const char* className)
{
  if (ShellCommandClass::commandClassList == NULL)
    initCommandClassList();

  tIterator<ShellCommandClass>* iterator = ShellCommandClass::commandClassList->getIterator();
  ShellCommandClass* elem = iterator->firstElement();
  while(elem != NULL)
  {
    if (!strcmp(className, elem->className))
    {
      delete iterator;
      return elem;
    }
    elem = iterator->nextElement();
  }
  delete iterator;
  return new ShellCommandClass(className);
}


void ShellCommandClass::initCommandClassList()
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    ShellCommandClass::commandClassList = new tList<ShellCommandClass>;
    ShellCommand<ShellCommandBase>::registerCommand("debug", "ShellCommand", &ShellCommandBase::debugDyn);
  }
}

tList<ShellCommandClass>* ShellCommandClass::commandClassList = NULL;

/**
 * constructs and registers a new Command
 * @param commandName the name of the Command
 * @param className the name of the class to apply this command to
 * @param paramCount the count of parameters this command takes
 * @return self
 */
ShellCommandBase::ShellCommandBase(const char* commandName, const char* className, unsigned int paramCount, ...)
{
  this->setClassID(CL_SHELL_COMMAND, "ShellCommand");
  this->setName(commandName);
  this->description = NULL;

//  this->classID = classID;
  ShellCommandClass::getCommandClass(className)->commandList->add(this); //ClassList::IDToString(classID);

  // handling parameters, and storing them:
  if (paramCount > FUNCTOR_MAX_ARGUMENTS)
    paramCount = FUNCTOR_MAX_ARGUMENTS;
  this->paramCount = paramCount;
  this->parameters = new unsigned int[paramCount];

  va_list parameterList;
  va_start(parameterList, paramCount);

  for (unsigned int i = 0; i < paramCount; i++)
  {
    this->parameters[i] = va_arg(parameterList, int);

    switch (this->parameters[i])
    {
      case ParameterBool:
        this->defaultBools[i] = va_arg(parameterList, int);
        break;
      case ParameterChar:
        this->defaultStrings[i] = new char[2];
        sprintf(this->defaultStrings[0], "%c",  va_arg(parameterList, int));
        break;
      case ParameterString:
        this->defaultStrings[i] = va_arg(parameterList, char*);
        break;
      case ParameterInt:
        this->defaultInts[i] = va_arg(parameterList, int);
        break;
      case ParameterUInt:
        this->defaultInts[i] = va_arg(parameterList, unsigned int);
        break;
      case ParameterFloat:
        this->defaultFloats[i] = va_arg(parameterList, double);
        break;
      case ParameterLong:
        this->defaultInts[i] = va_arg(parameterList, long);
        break;
      default:
        break;
    }
  }
}

/**
 * deconstructs a ShellCommand
 * @return
 */
ShellCommandBase::~ShellCommandBase()
{
  delete[] this->parameters;
}

/**
 * unregister an existing commandName
 * @param className the name of the Class the command belongs to.
 * @param commandName the name of the command itself
 *
 * @todo implement
 */
void ShellCommandBase::unregisterCommand(const char* commandName, const char* className)
{
  if (ShellCommandClass::commandClassList == NULL)
    ShellCommandClass::initCommandClassList();

 const  ShellCommandClass* checkClass = ShellCommandClass::isRegistered(className);

  if (checkClass != NULL)
  {
    tIterator<ShellCommandBase>* iterator = checkClass->commandList->getIterator();
    ShellCommandBase* elem = iterator->firstElement();
    while(elem != NULL)
    {
      if (!strcmp(commandName, elem->getName()))
      {
        checkClass->commandList->remove(elem);
        delete elem;
        break;
      }
      elem = iterator->nextElement();
    }
    delete iterator;

    if (checkClass->commandList->getSize() == 0)
    {
      ShellCommandClass::commandClassList->remove(checkClass);
      delete checkClass;
    }
  }
}



/**
 * checks if a command has already been registered.
 * @param commandName the name of the Command
 * @param className the name of the Class the command should apply to.
 * @param paramCount how many arguments the Command takes
 * @returns true, if the command is registered/false otherwise
 *
 * This is used internally, to see, if we have multiple command subscriptions.
 * This is checked in the registerCommand-function.
 */
bool ShellCommandBase::isRegistered(const char* commandName, const char* className, unsigned int paramCount, ...)
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    ShellCommandClass::initCommandClassList();
    return false;
  }

  const ShellCommandClass* checkClass = ShellCommandClass::isRegistered(className);
  if (checkClass != NULL)
  {
    tIterator<ShellCommandBase>* iterator = checkClass->commandList->getIterator();
    ShellCommandBase* elem = iterator->firstElement();
    while(elem != NULL)
   {
     if (!strcmp(commandName, elem->getName()))
     {
       PRINTF(2)("Command already registered\n");
       delete iterator;
       return true;
      }
     elem = iterator->nextElement();
   }
   delete iterator;
   return false;
  }
  else
    return false;
}


/**
 * executes commands
 * @param executionString the string containing the following input
 * ClassName [ObjectName] functionName [parameter1[,parameter2[,...]]]
 * @return true on success, false otherwise.
 */
bool ShellCommandBase::execute(const char* executionString)
{
//   if (ShellCommandBase::commandList == NULL)
//     return false;
//
//   tIterator<ShellCommandBase>* iterator = ShellCommandBase::commandList->getIterator();
//   ShellCommandBase* elem = iterator->firstElement();
//   while(elem != NULL)
//   {
//     printf("%s::%s\n", elem->className, elem->getName());
//     if (!strncasecmp (executionString, elem->className, strlen(elem->className)) &&
//          (*(executionString+strlen(elem->className)) == ' ' ||
//           *(executionString+strlen(elem->className)) == ':' ))
//     {
//       const char* commandBegin = executionString + strlen(elem->className);
//
//       PRINTF(4)("Class %s matches\n", elem->className);
//       BaseObject* objectPointer = NULL;
//       if (ClassList::StringToID(elem->className) & CL_MASK_SINGLETON == CL_MASK_SINGLETON)
//       {
//         while(*commandBegin == ' ')
//           commandBegin++;
//         if (strncmp (commandBegin, elem->getName(), strlen(elem->getName())) ||
//             *(commandBegin + strlen(elem->getName())) != ' ' &&
//             *(commandBegin + strlen(elem->getName())) != '\0')
//         {
//           elem = iterator->nextElement();
//           continue;
//         }
//         PRINTF(4)("Command %s matches\n", elem->getName());
//         // getting singleton-reference
//         tList<BaseObject>* list =  ClassList::getList(elem->className);
//         if (list != NULL)
//           objectPointer = list->firstElement();
//       }
//       else
//       {
//         // checking for the Object
//         while(*commandBegin == ' ')
//           commandBegin++;
//         tList<BaseObject>* list = ClassList::getList(elem->className);
//         if (list == NULL)
//           break;
//         tIterator<BaseObject>* iterBO = list->getIterator();
//         BaseObject* enumBO = iterBO->firstElement();
//         while(enumBO != NULL)
//         {
//           if(!strncmp(commandBegin, enumBO->getName(), strlen(enumBO->getName())))
//           {
//             PRINTF(4)("Object %s matches\n", enumBO->getName());
//             objectPointer = enumBO;
//             break;
//           }
//           enumBO = iterBO->nextElement();
//         }
//         delete iterBO;
//
//         // break on no object Found. We cannot operate on Classes, but on Objects
//         if (objectPointer == NULL)
//           break;
//         commandBegin = commandBegin + strlen(objectPointer->getName());
//         while(*commandBegin == ' ')
//           commandBegin++;
//         // checking for the requested function.
//         if (strncmp (commandBegin, elem->getName(), strlen(elem->getName())))
//         {
//           elem = iterator->nextElement();
//           continue;
//         }
//         PRINTF(4)("Function '%s' found\n", commandBegin);
//       }
//       const char* paramBegin = strchr(commandBegin, ' ');
//       if (paramBegin == NULL)
//         paramBegin = commandBegin + strlen(elem->getName());
//       while (*paramBegin == ' ')
//         paramBegin++;
//
//       PRINTF(3)("Parameters to Pass: %s\n", paramBegin);
//       if (objectPointer != NULL && paramBegin != NULL)
//       {
//         elem->executeCommand(objectPointer, paramBegin);
//         delete iterator;
//         return true;
//       }
//     }
//     elem = iterator->nextElement();
//   }
//   delete iterator;
//   return true;
}

/**
 * lets a command be described
 * @param description the description of the Given command
 */
ShellCommandBase* ShellCommandBase::describe(const char* description)
{
  if (this == NULL)
    return NULL;
 else
 {
   this->description = description;
   return this;
 }
}

/**
 * see ShellCommandBase::debug()
 */
void ShellCommandBase::debugDyn()
{
  this->debug();
}

/**
 * prints out nice information about the Shells Commands
 */
void ShellCommandBase::debug()
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    PRINT(0)("No Command registered.\n");
    return;
  }

  tIterator<ShellCommandClass>* iteratorCL = ShellCommandClass::commandClassList->getIterator();
  ShellCommandClass* elemCL = iteratorCL->firstElement();
  while(elemCL != NULL)
  {
    PRINT(0)("Class:'%s' registered %d commands: \n", elemCL->className, elemCL->commandList->getSize());
    tIterator<ShellCommandBase>* iterator = elemCL->commandList->getIterator();
    const ShellCommandBase* elem = iterator->firstElement();
    {
      PRINT(0)("  command:'%s' : params:%d: ", elem->getName(), elem->paramCount);
      for (unsigned int i = 0; i< elem->paramCount; i++)
       printf("%s ", ShellCommandBase::paramToString(elem->parameters[i]));
      if (elem->description != NULL)
       printf("- %s", elem->description);
      printf("\n");

      elem = iterator->nextElement();
    }
    delete iterator;
    elemCL = iteratorCL->nextElement();
  }
  delete iteratorCL;
}

/**
 * converts a Parameter to a String
 * @param parameter the Parameter we have.
 * @returns the Name of the Parameter at Hand
 */
const char* ShellCommandBase::paramToString(long parameter)
{
  switch (parameter)
  {
    case ParameterBool:
      return "BOOL";
      break;
    case ParameterChar:
      return "CHAR";
      break;
    case ParameterString:
      return "STRING";
      break;
    case ParameterInt:
      return "INT";
      break;
    case ParameterUInt:
      return "UINT";
      break;
    case ParameterFloat:
      return "FLOAT";
      break;
    case ParameterLong:
      return "LONG";
      break;
    default:
      return "NULL";
      break;
  }
}
