/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

/*!
 * @file load_param.h
 * A Class and macro-functions, that makes our lives easy to load-in parameters
 */

#ifndef _LOAD_PARAM_H
#define _LOAD_PARAM_H

#include "base_object.h"

#include "factory.h"
#include "debug.h"
#include "substring.h"
#include "tinyxml.h"

// Forward Declaration //
template<class T> class tList;

//! macro that makes it even more easy to load a Parameter
/**
 * @param className the name of the class to load
 * @param parameterName the name of the parameter to load as written in the XML-file
 * @param function the function to call
 */
#define LOAD_PARAM(className, parameterName, paramFunction) \
        LoadParam<className>(root, #parameterName, this, &className::paramFunction)

/**
 * this Starts a Cycle in the Loading Process
 * be aware, that in the cycle the first parameter of load_param should because
 * called element, and that you must say true at the Fith parameter, or it will fail
 * also you will have to close the Cycle again with LOAD_PARAM_END_CYCLE
 */
#define LOAD_PARAM_START_CYCLE   const TiXmlElement* element; \
                                 element = root->FirstChildElement(); \
                                 while( element != NULL) \
                                  {
/**
 * closes a LoadParam Loop
 * @see LOAD_PARAM_START_CYCLE
 */
#define LOAD_PARAM_END_CYCLE        element = element->NextSiblingElement(); \
                                  }



/**
   useable FunctionParameters are:
   l_INT:       int
   l_LONG:      long
   l_SHORT:     short
   l_FLOAT:     float
   l_STRING:    const char*
   l_XML_ELEM:  TiXmlElement*
*/

#define l_BOOL_TYPE        bool                 //!< The type of an BOOL
#define l_BOOL_FUNC        isBool               //!< The function to call to parse BOOL
#define l_BOOL_NAME        "bool"               //!< The name of an BOOL
#define l_BOOL_DEFAULT     false                //!< a default Value for an BOOL


#define l_INT_TYPE         int                  //!< The type of an INT
#define l_INT_FUNC         isInt                //!< The function to call to parse INT
#define l_INT_NAME         "int"                //!< The name of an INT
#define l_INT_DEFAULT      0                    //!< a default Value for an INT

#define l_UINT_TYPE        unsigned int         //!< The type of an UINT
#define l_UINT_FUNC        isInt                //!< The function to call to parse UINT
#define l_UINT_NAME        "unsigned int"       //!< The name of an UINT
#define l_UINT_DEFAULT     0                    //!< a default Value for an UINT

#define l_LONG_TYPE        long                 //!< The type of a LONG
#define l_LONG_FUNC        isInt                //!< The function to parse a LONG
#define l_LONG_NAME        "long"               //!< The name of a LONG
#define l_LONG_DEFAULT     0                    //!< a default Value for a LONG

// #define l_SHORT_TYPE       short                //!< The type of a SHORT
// #define l_SHORT_FUNC       atoi                 //!< The function to parse a SHORT
// #define l_SHORT_NAME       "short"              //!< The name of a SHORT
// #define l_SHORT_DEFAULT    0                    //!< a default Value for a SHORT

#define l_FLOAT_TYPE       float                //!< The type of a FLOAT
#define l_FLOAT_FUNC       isFloat              //!< The function to parse a FLOAT
#define l_FLOAT_NAME       "float"              //!< The name of a FLOAT
#define l_FLOAT_DEFAULT    0.0                  //!< a default Value for a FLOAT

//#define l_VECTOR_TYPE      const Vector&        //!< The type of a VECTOR
//#define l_VECTOR_FUNC      isVector             //!< The function to parse a VECTOR
//#define l_VECTOR_NAME      "Vector[x/y/z]"      //!< The name of a VECTOR
//#define l_VECTOR_DEFAULT   Vector(0,0,0)        //!< Default value for a VECTOR

#define l_STRING_TYPE      const char*          //!< The type of a STRING
#define l_STRING_FUNC      isString             //!< The function to parse a STRING
#define l_STRING_NAME      "string"             //!< The name of a STRING
#define l_STRING_DEFAULT   ""                   //!< a default Value for an STRING

#define l_XML_ELEM_TYPE    const TiXmlElement*  //!< The type of an XML_ELEM
#define l_XML_ELEM_FUNC                         //!< The function to parse an XML_ELEM
#define l_XML_ELEM_NAME    "XML"                //!< The name of an XML_ELEM
#define l_XML_ELEM_DEFAULT NULL                 //!< The dafault Value for an XML_ELEM


/*****************************************
**** MACROS DEFINITIONS OF LOADABLES *****
*****************************************/
// 1. TYPE
/**
 *  a Macro to easily implement many different Constructors for the LoadParam-Class with 1 argument
 * @param type1 The type of the first functionParameter
*/
#define LoadParam1(type1) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE), \
           bool multi = false, type1##_TYPE default1 = type1##_DEFAULT) \
  : BaseLoadParam(root, pt2Object, paramName, 1, multi, NULL, type1##_NAME, default1) \
    { \
      if (loadString != NULL && root != NULL) \
        (*pt2Object.*function)(type1##_FUNC(loadString, default1)); \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
    }

// 2. TYPES
/**
 *  a Macro to easily implement many different Constructors for the LoadParam-Class with 2 arguments
 * @param type1 The type of the first functionParameter
 * @param type2 The type of the second functionParameter
*/
#define LoadParam2(type1, type2) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE, type2##_TYPE), \
           bool multi = false,  type1##_TYPE default1 = type1##_DEFAULT,  type2##_TYPE default2 = type2##_DEFAULT) \
  : BaseLoadParam(root, pt2Object, paramName, 2, multi, NULL, type1##_NAME, default1, type2##_NAME, default2) \
    { \
      if (loadString != NULL && root != NULL) \
        { \
          SubString subLoads(loadString); \
          if (subLoads.getCount() == 2) \
            (*pt2Object.*function)(type1##_FUNC(subLoads.getString(0), default1), type2##_FUNC(subLoads.getString(1), default2)); \
          else \
            PRINTF(2)("Not loaded Parameter %s of %s, because wrong count of arguments.\n -> Should have %d but have %d\n", \
                      paramName, pt2Object->getClassName(), 2, subLoads.getCount()); \
        } \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
    }


// 3. TYPES
/**
 *  a Macro to easily implement many different Constructors for the LoadParam-Class with 3 arguments
 * @param type1 The type of the first functionParameter
 * @param type2 The type of the second functionParameter
 * @param type3 The type of the third functionParameter
*/
#define LoadParam3(type1, type2, type3) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE, type2##_TYPE, type3##_TYPE), \
           bool multi = false, type1##_TYPE default1 = type1##_DEFAULT, type2##_TYPE default2 = type2##_DEFAULT, type3##_TYPE default3 = type3##_DEFAULT)\
  : BaseLoadParam(root, pt2Object, paramName, 3, multi, NULL, type1##_NAME, default1, type2##_NAME, default2, type3##_NAME, default3) \
    { \
      if (loadString != NULL && root != NULL) \
        { \
          SubString subLoads(loadString); \
          if (subLoads.getCount() == 3) \
            (*pt2Object.*function)(type1##_FUNC(subLoads.getString(0), default1), type2##_FUNC(subLoads.getString(1), default2), type3##_FUNC(subLoads.getString(2), default3)); \
          else \
            PRINTF(2)("Not loaded Parameter %s of %s, because wrong count of arguments.\n -> Should have %d but have %d\n", \
                      paramName, pt2Object->getClassName(), 3, subLoads.getCount()); \
        } \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
    }


// 4. TYPES
/**
 *  a Macro to easily implement many different Constructors for the LoadParam-Class with 4 arguments
 * @param type1 The type of the first functionParameter
 * @param type2 The type of the second functionParameter
 * @param type3 The type of the third functionParameter
 * @param type4 The type of the forth functionParameter
*/
#define LoadParam4(type1, type2, type3, type4) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE, type2##_TYPE, type3##_TYPE, type4##_TYPE), \
           bool multi = false, type1##_TYPE default1 = type1##_DEFAULT, type2##_TYPE default2 = type2##_DEFAULT, type3##_TYPE default3 = type3##_DEFAULT, \
           type4##_TYPE default4 = type4##_DEFAULT) \
  : BaseLoadParam(root, pt2Object, paramName, 4, multi, NULL, type1##_NAME, default1, type2##_NAME, default2, type3##_NAME, default3, \
                  type4##_NAME, default4) \
    { \
      if (loadString != NULL && root != NULL) \
        { \
          SubString subLoads(loadString); \
          if (subLoads.getCount() == 4) \
            (*pt2Object.*function)(type1##_FUNC(subLoads.getString(0), default1), type2##_FUNC(subLoads.getString(1), default2), type3##_FUNC(subLoads.getString(2), default3), type4##_FUNC(subLoads.getString(3), default4)); \
          else \
            PRINTF(2)("Not loaded Parameter %s of %s, because wrong count of arguments.\n -> Should have %d but have %d\n", \
                      paramName, pt2Object->getClassName(), 4, subLoads.getCount()); \
        } \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
    }


// 5. TYPES
/**
 *  a Macro to easily implement many different Constructors for the LoadParam-Class with 5 arguments
 * @param type1 The type of the first functionParameter
 * @param type2 The type of the second functionParameter
 * @param type3 The type of the third functionParameter
 * @param type4 The type of the forth functionParameter
 * @param type5 The type of the fifth functionParameter
*/
#define LoadParam5(type1, type2, type3, type4, type5) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, \
           void(T::*function)(type1##_TYPE, type2##_TYPE, type3##_TYPE, type4##_TYPE, type5##_TYPE), \
           bool multi = false, type1##_TYPE default1 = type1##_DEFAULT, type2##_TYPE default2 = type2##_DEFAULT, type3##_TYPE default3 = type3##_DEFAULT, \
           type4##_TYPE default4 = type4##_DEFAULT, type5##_TYPE default5 = type5##_DEFAULT ) \
  : BaseLoadParam(root, pt2Object, paramName, 5, multi, NULL, type1##_NAME, default1, type2##_NAME, default2, type3##_NAME, default3, \
                  type4##_NAME, default4, type5##_NAME, default5) \
    { \
      if (loadString != NULL && root != NULL) \
        { \
          SubString subLoads(loadString); \
          if (subLoads.getCount() == 5) \
            (*pt2Object.*function)(type1##_FUNC(subLoads.getString(0), default1), type2##_FUNC(subLoads.getString(1), default2), type3##_FUNC(subLoads.getString(2), default3), type4##_FUNC(subLoads.getString(3), default4), type5##_FUNC(subLoads.getString(4), default5)); \
          else \
            PRINTF(2)("Not loaded Parameter %s of %s, because wrong count of arguments.\n -> Should have %d but have %d\n", \
                      paramName, pt2Object->getClassName(), 5, subLoads.getCount()); \
        } \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
    }

// Pointer TYPE
/**
 *  a Macro to easily implement many different Constructors for the LoadParam-Class with one Pointer argument
 * @param type1 The type of the Pointer
 */
#define LoadParamPT(type1) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE), type1##_TYPE pointerToParam, bool multi = false) \
  : BaseLoadParam(root, pt2Object, paramName, 1, multi, pointerToParam, type1##_NAME) \
{ \
      if (pointerToParam != NULL && root != NULL) \
        (*pt2Object.*function)((type1##_TYPE) pointerToParam); \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
}


/***********************
*** HELPER FUNCTIONS ***
***********************/
bool          isBool(const char* BOOL, bool defaultValue);
int           isInt(const char* INT, int defaultValue);
float         isFloat(const char* FLOAT, float defaultValue);
//const Vector& isVector(const char* VECTOR, const Vector& defaultValue);
const char*   isString(const char* STRING, const char* defaultValue);

//TiXmlEmlemnt* isXmlElem(const)


/************************
*** DESCRIPTION STUFF ***
************************/
//! A class that handles the description of loadable parameters
class LoadParamDescription
{
  friend class BaseLoadParam;
  friend class LoadClassDescription;
 public:
  LoadParamDescription(const char* paramName);
  ~LoadParamDescription();

  void setDescription(const char* descriptionText);
  /** @returns the descriptionString */
  const char* getDescription() { return this->description; };

  void print() const;
 private:
  char*         paramName;             //!< The name of the parameter.
  int           paramCount;            //!< The count of parameters.
  char**        types;                 //!< What kind of parameters does this function take ??
  char*         description;           //!< A longer description about this function.
  char**        defaultValues;         //!< The 'Default Values'.
};

//! A class for descriptions of a loadable module
class LoadClassDescription
{
  friend class BaseLoadParam;
 public:
  LoadClassDescription(const char* className);
  ~LoadClassDescription();

  static LoadClassDescription* addClass(const char* className);
  LoadParamDescription* addParam(const char* paramName);

  static void printAll(const char* fileName = NULL);
  static tList<const char>* searchClassWithShort(const char* classNameBegin);
//  static const LoadParamDescription* getClass(const char* className);

 private:
  static bool                          parametersDescription;  //!< if parameter-description should be enabled.
  static tList<LoadClassDescription>*  classList;              //!< a list, that holds all the loadable classes. (after one instance has been loaded)
  char*                                className;              //!< name of the class
  tList<LoadParamDescription>*         paramList;              //!< List of parameters this class knows.
};


/**************************
**** REAL DECLARATIONS ****
**************************/
//! abstract Base class for a Loadable parameter
class BaseLoadParam : public BaseObject
{
 public:
  BaseLoadParam* describe(const char* descriptionText);

 protected:
  BaseLoadParam(const TiXmlElement* root, BaseObject* object, const char* paramName, int paramCount, bool multi, const void* pointerToParam, ...);

 protected:
  LoadClassDescription*    classDesc;            //!< The LoadClassDescription of this LoadParameter
  LoadParamDescription*    paramDesc;            //!< The LoadParameterDescription of this LoadParameter
  const char*              loadString;           //!< The string loaded by this LoadParam
  const void*              pointerToParam;       //!< A Pointer to a Parameter.
};


//! derived template class, so all the Classes can load something.
template<class T> class LoadParam : public BaseLoadParam
{
 public:
  LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(), bool multi = false)
    : BaseLoadParam(root, pt2Object, paramName, 0, multi, NULL, "")
    {
      if (loadString != NULL && root != NULL)
        (*pt2Object.*function)();
      else
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName());
    }


#define FUNCTOR_LIST(x)    LoadParam ## x
#include "functor_list.h"
#undef FUNCTOR_LIST

  //! makes functions with one Vector loadable
  //LoadParam1(l_VECTOR);

  // loads a Ti-XML-element
  LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(const TiXmlElement*), bool multi = false)
  : BaseLoadParam(root, pt2Object, paramName, 1, multi, NULL, "XML")
  {
    if (root != NULL)
    {
      const TiXmlElement* elem = root->FirstChildElement(paramName);
      if (likely(elem != NULL))
        (*pt2Object.*function)(elem);
      else
        PRINTF(2)("%s of %s is empty", paramName, pt2Object->getClassName());
    }
    else
      PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName());
  }

  //LoadParamPT(l_XML_ELEM);
};

// helper function

const char* grabParameter(const TiXmlElement* root, const char* parameterName);

#endif /* _LOAD_PARAM_H */
