/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: ...
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "render_2d.h"

#include "graphics_engine.h"
#include "class_list.h"
#include "list.h"
#include "element_2d.h"

#include <math.h>

using namespace std;



/**
 * standard constructor
 */
Render2D::Render2D ()
{
   this->setClassID(CL_RENDER_2D, "Render2D");
   this->setName("Render2D");

   for (int i = 0; i < E2D_LAYER_COUNT; i++)
     this->element2DList[i] = new tList<Element2D>;
}

/**
 *  the singleton reference to this class
 */
Render2D* Render2D::singletonRef = NULL;

/**
 * standard deconstructor
 */
Render2D::~Render2D ()
{
  for (int i = 0; i < E2D_LAYER_COUNT; i++)
    delete this->element2DList[i];

  delete NullElement2D::getInstance();
  Render2D::singletonRef = NULL;
}


/**
 * registers a 2D-element to the 2D-Renderer
 * @param element2D the element to registers
 *
 * do not use this function by yourself, because this is used by Element2D's constructor
 */
void Render2D::registerElement2D(Element2D* element2D)
{
  this->element2DList[(int)log2(E2D_DEFAULT_LAYER)]->add(element2D);
}

/**
 * unregisters a 2D-element from the 2D-Renderer
 * @param element2D The element to unregister
 *
 * do not use this function by yourself, because this is used by Element2D's destructor
 */
void Render2D::unregisterElement2D(Element2D* element2D)
{
  this->element2DList[(int)log2(element2D->getLayer())]->remove(element2D);
}


/**
 * moves an 2D-Element to another Layer
 * @param element2D the Element to move
 * @param to which layer to move it to.
 */
void Render2D::moveToLayer(Element2D* element2D, E2D_LAYER to)
{
  if (element2D->getLayer() != to)
  {
    this->element2DList[(int)log2(element2D->getLayer())]->remove(element2D);
    this->element2DList[(int)log2(to)]->add(element2D);
  }
}


/**
 * ticks all the 2d-elements
 * @param dt the timestep since last dt
 */
void Render2D::tick(float dt)
{
  for (int i = 0; i < E2D_LAYER_COUNT; i++)
  {
    tIterator<Element2D>* iterator = this->element2DList[i]->getIterator();
    Element2D* elem = iterator->firstElement();
    while (elem != NULL)
    {
      if (elem->isActive())
        elem->tick(dt);
      elem = iterator->nextElement();
    }
    delete iterator;
  }
}

/**
 * renders all the Elements of the Render2D-engine
 * @param layer the Layer to draw
 */
void Render2D::draw(unsigned int layer) const
{
  GraphicsEngine::enter2DMode();

  int drawLayer = 1;

  for (int i = 0; i < E2D_LAYER_COUNT; i++)
  {
    if (layer & drawLayer && this->element2DList[i]->getSize() > 0)
    {
      tIterator<Element2D>* iterator = this->element2DList[i]->getIterator();
      Element2D* elem = iterator->firstElement();
      while (elem != NULL)
      {
        if (elem->isVisible())
          elem->draw();
        elem = iterator->nextElement();
      }
      delete iterator;
    }
    drawLayer << 1;
  }
  GraphicsEngine::leave2DMode();

}
