/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#include "load_param.h"

#include "list.h"
#include "base_object.h"

#include <stdarg.h>

/**
 * @param object The object this Parameter is loaded too.
 * @param root: the XML-element to load this option from.
 * @param paramName: The name of the parameter loaded.
 * @param paramCount: how many parameters this loading-function takes
 * @param multi: if false LoadParam assumes only one occurence of this parameter in root, if true it assumes multiple occurences.
 * @param ...: the parameter information (1. Parameter, 2. Default Value for the Parameter, ...)
*/
BaseLoadParam::BaseLoadParam(const TiXmlElement* root, BaseObject* object, const char* paramName,
                             int paramCount, bool multi, const void* pointerToParam, ...)
{
  this->setClassID(CL_LOAD_PARAM, "LoadParam");

  this->loadString = NULL;
  this->pointerToParam = pointerToParam;

  if (paramCount == 0 || this->pointerToParam)
    this->loadString = "none";
  else
    {
      if (likely(!multi))
        this->loadString = grabParameter(root, paramName);
      else
        {
          if (!strcmp(root->Value(), paramName))
            {
              const TiXmlNode* val = root->FirstChild();
              if( val->ToText())
                this->loadString = val->Value();
            }
        }
    }

  this->paramDesc = NULL;
  if (LoadClassDescription::parametersDescription)
  {
    // locating the class
    this->classDesc = LoadClassDescription::addClass(object->getClassName());

    if ((this->paramDesc = this->classDesc->addParam(paramName)) != NULL)
    {

      this->paramDesc->paramCount = paramCount;
      this->paramDesc->types = new char*[paramCount];
      this->paramDesc->defaultValues = new char*[paramCount];

      va_list types;
      va_start (types, pointerToParam);
      char defaultVal[512];
      for(int i = 0; i < paramCount; i++)
      {
          // parameters parsed
        const char* tmpTypeName = va_arg (types, const char*);
        this->paramDesc->types[i] = new char[strlen(tmpTypeName)+1];
        strcpy(this->paramDesc->types[i], tmpTypeName);

          // default value description
        if (!strcmp(tmpTypeName, l_INT_NAME))
        {
          sprintf(defaultVal, "%d", va_arg(types, l_INT_TYPE));
        }
        else if (!strcmp(tmpTypeName, l_LONG_NAME))
        {
          sprintf(defaultVal, "%0.3f", va_arg(types, l_LONG_TYPE));
        }
          /*          else if (!strcmp(tmpTypeName, l_SHORT_NAME))
        {
        sprintf(defaultVal, "%d", va_arg(types, l_SHORT_TYPE));
      }*/
        else if (!strcmp(tmpTypeName, l_FLOAT_NAME))
        {
          sprintf(defaultVal, "%0.3f", va_arg(types, double));
        }
        else if (!strcmp(tmpTypeName, l_STRING_NAME))
        {
          sprintf(defaultVal, "%s", va_arg(types, l_STRING_TYPE));
        }
        else if (!strcmp(tmpTypeName, l_XML_ELEM_NAME))
        {
          sprintf(defaultVal, "");
        }

        this->paramDesc->defaultValues[i] = new char[strlen(defaultVal)+1];
        strcpy(this->paramDesc->defaultValues[i], defaultVal);
      }
      va_end(types);

      int argCount = 0;
    }
  }
}

/**
 * checks if the input was a Bool
 * @param BOOL a String that holds a bool: must be one of those: 1,0,true,false,TRUE,FALSE
 * @param defaultValue a default value that is set, if BOOL is corrupt
 * @return returns the bool, if BOOL was correct otherwise defaultValue
 */
bool isBool(const char* BOOL, bool defaultValue)
{
  if(!strcmp(BOOL, "1") || !strcmp( BOOL,"true") || !strcmp(BOOL,"TRUE"))
    return true;
  else if (!strcmp(BOOL, "0") || !strcmp( BOOL,"false") || !strcmp(BOOL,"FALSE"))
    return false;
  else
    return defaultValue;

}

int isInt(const char* INT, int defaultValue)
{
  char* endPtr = NULL;
  int result = strtol(INT, &endPtr, 10);

  if ( endPtr >= INT && endPtr < INT + strlen(INT))
    return defaultValue;
  else
    return result;
}

float isFloat(const char* FLOAT, float defaultValue)
{
  char* endPtr = NULL;
  double result = strtod(FLOAT, &endPtr);

  if ( endPtr >= FLOAT && endPtr < FLOAT + strlen(FLOAT))
    return defaultValue;
  else
    return result;
}

const Vector& isVector(const char* VECTOR, const Vector& defaultValue)
{


}

const char* isString(const char* STRING, const char* defaultValue)
{
  if (STRING != NULL)
    return STRING;
  else
    return defaultValue;
}


/**
 * @param descriptionText The text to set as a description for this Parameter
 * @returns a pointer to itself.
*/
BaseLoadParam* BaseLoadParam::describe(const char* descriptionText)
{
  if (LoadClassDescription::parametersDescription && this->paramDesc && !this->paramDesc->getDescription())
    {
      this->paramDesc->setDescription(descriptionText);
    }
  return this;
}

/**
 * @param paramName the name of the parameter to load
*/
LoadParamDescription::LoadParamDescription(const char* paramName)
{
  this->types = NULL;
  this->description = NULL;
  this->paramName = new char[strlen(paramName)+1];
  strcpy(this->paramName, paramName);
  this->defaultValues = NULL;
}

/**
 *  removes all the alocated memory
*/
LoadParamDescription::~LoadParamDescription()
{
  if (this->types != NULL)
  {
    for(int i = 0; i < this->paramCount; i++)
    {
      delete []this->types[i];
    }
  }
  if (this->defaultValues != NULL)
  {
    for(int i = 0; i < this->paramCount; i++)
    {
      delete []this->defaultValues[i];
    }
  }

  delete []this->types;
  delete []this->defaultValues;
  delete []this->paramName;
  delete []this->description;
}

/**
 * @param descriptionText The text to set as a description for this Parameter
*/
void LoadParamDescription::setDescription(const char* descriptionText)
{
  this->description = new char[strlen(descriptionText)+1];
  strcpy(this->description, descriptionText);
}

/**
 *  prints out this parameter, its input method and the description (if availiable)
*/
void LoadParamDescription::print() const
{
  PRINT(3)(" <%s>", this->paramName);
  for (int i = 0; i < this->paramCount; i++)
    {
      if (i > 0)
        PRINT(3)(",");
      PRINT(3)("%s", this->types[i]);
    }
  PRINT(3)("</%s>", this->paramName);
  if (this->description)
    PRINT(3)(" -- %s", this->description);
  // default values
  if (this->paramCount > 0)
  {
    PRINT(3)(" (Default: ");
    for (int i = 0; i < this->paramCount; i++)
    {
      if (i > 0)
        PRINT(3)(", ");
      if (!strcmp(this->types[i], l_STRING_NAME))
      { // leave brackets !!
        PRINT(3)("\"%s\"", this->defaultValues[i]);
      }
      else
      {
        PRINT(3)("%s", this->defaultValues[i]);
      }
    }
    PRINT(3)(")");
  }
  PRINT(3)("\n");
}

/**
 *  A list, that holds all the classes that are loadable (classes not objects!!)
*/
tList<LoadClassDescription>* LoadClassDescription::classList = new tList<LoadClassDescription>;

/**
 *  if the description of Parameters should be executed
*/
bool LoadClassDescription::parametersDescription = true;

/**
 * @param className the name of the class to be loadable
*/
LoadClassDescription::LoadClassDescription(const char* className)
{
  this->className = new char[strlen(className)+1];
  strcpy(this->className, className);

  classList->add(this);

  this->paramList = new tList<LoadParamDescription>;
}

/**
 *  deletes a classDescription (deletes all the parameterDescriptions as well
*/
LoadClassDescription::~LoadClassDescription()
{
  delete []this->className;

  tIterator<LoadParamDescription>* iterator = this->paramList->getIterator();
  LoadParamDescription* enumParamDesc = iterator->firstElement();
  while (enumParamDesc)
    {
      delete enumParamDesc;
      enumParamDesc = iterator->nextElement();
    }
  delete iterator;
}

/**
 *  adds a class to the list of loadable classes
 * @param className The name of the class to add

   this function searches for the className string, and if found just returns the appropriate Class.
   Otherwise it returns a new classDescription
*/
LoadClassDescription* LoadClassDescription::addClass(const char* className)
{
  tIterator<LoadClassDescription>* iterator = LoadClassDescription::classList->getIterator();
  LoadClassDescription* enumClassDesc = iterator->firstElement();
  while (enumClassDesc)
    {
      if (!strcmp(enumClassDesc->className, className))
        {
          delete iterator;
          return enumClassDesc;
        }
      enumClassDesc = iterator->nextElement();
    }
  delete iterator;

  return new LoadClassDescription(className);
}

/**
 *  does the same as addClass(const char* className), but with params
 * @param paramName the name of the parameter to add.
*/
LoadParamDescription* LoadClassDescription::addParam(const char* paramName)
{
  tIterator<LoadParamDescription>* iterator = this->paramList->getIterator();
  LoadParamDescription* enumParamDesc = iterator->firstElement();
  while (enumParamDesc)
    {
      if (!strcmp(enumParamDesc->paramName, paramName))
        {
          delete iterator;
          return enumParamDesc;
        }
      enumParamDesc = iterator->nextElement();
    }
  delete iterator;

  this->paramList->add(new LoadParamDescription(paramName));
  return paramList->lastElement();
}

/**
 *  prints out all loadable Classes, and their parameters
 * @param fileName prints the output to a File
 * @todo implement it
*/
void LoadClassDescription::printAll(const char* fileName)
{
  PRINT(3)("===============================================================\n");
  PRINT(3)(" Listing all the Loadable Options (loaded since Game started).\n\n");
  tIterator<LoadClassDescription>* classIT = LoadClassDescription::classList->getIterator();
  LoadClassDescription* enumClassDesc = classIT->firstElement();
  while (enumClassDesc)
    {
      PRINT(3)("<%s>\n", enumClassDesc->className);
      tIterator<LoadParamDescription>* paramIT = enumClassDesc->paramList->getIterator();
      LoadParamDescription* enumParamDesc = paramIT->firstElement();
      while (enumParamDesc)
        {
          enumParamDesc->print();
          enumParamDesc = paramIT->nextElement();
        }
      delete paramIT;

      PRINT(3)("</%s>\n\n", enumClassDesc->className);
      enumClassDesc = classIT->nextElement();
    }
  delete classIT;
  PRINT(3)("===============================================================\n");
}

/**
 * searches for classes, which beginn with classNameBegin
 * @param classNameBegin the beginning string of a Class
 * @return a NEW char-array with ClassNames. The LIST should be deleted afterwards,
 * !! The strings MUST NOT be deleted !!
 */
tList<const char>* LoadClassDescription::searchClassWithShort(const char* classNameBegin)
{
  unsigned int searchLength = strlen(classNameBegin);
  tList<const char>* retVal = new tList<const char>;

  tIterator<LoadClassDescription>* iterator = LoadClassDescription::classList->getIterator();
  LoadClassDescription* enumClassDesc = iterator->firstElement();
  while (enumClassDesc)
  {
    if (strlen(enumClassDesc->className)>searchLength+1 &&
        !strncasecmp(enumClassDesc->className, classNameBegin, searchLength))
    {
      retVal->add(enumClassDesc->className);
    }
    enumClassDesc = iterator->nextElement();
  }
  delete iterator;

  return retVal;
}

// const LoadParamDescription* LoadParamDescription::getClass(const char* className)
// {
//   tIterator<LoadClassDescription>* iterator = LoadClassDescription::classList->getIterator();
//   LoadClassDescription* enumClassDesc = iterator->firstElement();
//   while (enumClassDesc)
//   {
//     if (!strcmp(enumClassDesc->className, classNameBegin, className))
//     {
//       delete iterator;
//       return enumClassDesc;
//     }
//     enumClassDesc = iterator->nextElement();
//   }
//   delete iterator;
//
//   return NULL;
// }

/**
 * @param root: The XML-element to grab a parameter from
 * @param parameterName: the parameter to grab
 * @returns the Value of the parameter if found, NULL otherwise
*/
const char* grabParameter(const TiXmlElement* root, const char* parameterName)
{
  const TiXmlElement* element;
  const TiXmlNode* node;

  if (root == NULL)
    return NULL;
  assert( parameterName != NULL);

  element = root->FirstChildElement( parameterName);
  if( element == NULL) return NULL;

  node = element->FirstChild();
  while( node != NULL)
    {
      if( node->ToText()) return node->Value();
      node = node->NextSibling();
    }
  return NULL;
}
