/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_ANIM

#include "animation_player.h"

#include "compiler.h"

using namespace std;


/**
 *  standard constructor
*/
AnimationPlayer::AnimationPlayer ()
{
  this->setClassID(CL_ANIMATION_PLAYER, "AnimationPlayer");
  this->setName("AnimationPlayer");

  this->animationList = new tList<Animation>();
  this->play();
}

/**
 *  the singleton reference to this class
*/
AnimationPlayer* AnimationPlayer::singletonRef = NULL;

/**
 *  standard deconstructor

   !! DANGER !! when unloading the AnimationPlayer no other Function
   should reference any Animations, from the animationList because it
   automatically deletes them.
   This usually happens when unloading a World.
*/
AnimationPlayer::~AnimationPlayer ()
{
  // deleting the Animation List AND all the elements of the List
  this->flush();
  delete this->animationList;

  AnimationPlayer::singletonRef = NULL;
}

/**
 *  adds an Animation to the AnimationList.
 * @param animation the Animation to handle

   when adding a Animation the Animation will too be deleted when
   the AnimationPlayer gets deleted. Consider not adding it, or
   unadding it with animation->notHandled();
*/
void AnimationPlayer::addAnimation(Animation* animation)
{
  this->animationList->add(animation);
}

/**
 *  removes an Animation from the Animation List, WITHOUT deleting it.
 * @param animation the Anmination to remove from the List
*/
void AnimationPlayer::removeAnimation(Animation* animation)
{
  this->animationList->remove(animation);
}

/**
 *  empties the list AND deletes all the Animations
*/
void AnimationPlayer::flush()
{
  // deleting the Animation List AND all the elements of the List
  tIterator<Animation>* animIt = this->animationList->getIterator();
  Animation* anim = animIt->firstElement();
  while( anim != NULL)
    {
      delete anim;
      this->animationList->remove(anim);
      anim = animIt->nextElement();
    }
  delete animIt;

  delete this->animationList;
  this->animationList = new tList<Animation>();
}

/**
 *  Ticks all the animations in animationList
 * @param timePassed the time passed since the last tick.
*/
void AnimationPlayer::tick(float timePassed)
{
  if (this->bRunning)
    {
      // iterate through all the animations and tick them.
      tIterator<Animation>* animIt = this->animationList->getIterator();
      Animation* anim = animIt->firstElement();
      while( anim != NULL)
        {
          anim->tick(timePassed);
          if(unlikely(anim->ifDelete()))
          {
            this->animationList->remove(anim);
            delete anim;
          }
          anim = animIt->nextElement();
        }
      delete animIt;
    }
}
/**
 *  starts playing the AnimationPlayer
*/
void AnimationPlayer::play()
{
  this->bRunning = true;
}

/**
 *  pauses playing of the AnimationPlayer
*/
void AnimationPlayer::pause()
{
  this->bRunning = false;
}

/**
 * @returns the animation from a certain baseobject
   if multiple are found, it just retruns the first one
   if none is found it returns NULL
*/
Animation* AnimationPlayer::getAnimationFromBaseObject(const BaseObject* baseObject) const
{
  tIterator<Animation>* animIt = this->animationList->getIterator();
  Animation* anim = animIt->firstElement();
  while( anim != NULL)
    {
      if(anim->getBaseObject() == baseObject)
        {
          delete animIt;
          return anim;
        }
      anim = animIt->nextElement();
    }
  delete animIt;
  return NULL;
}



/**
 *  Outputs some nice debug-information
*/
void AnimationPlayer::debug()
{
  PRINT(0)("+------------------------------------+\n");
  PRINT(0)("+ ANIMATION PLAYER DEBUG INFORMATION +\n");
  PRINT(0)("+------------------------------------+\n");
  PRINT(0)("| Reference: %p\n", this);
  PRINT(0)("| CountOfAnims %d\n", this->animationList->getSize());
  PRINT(0)("-Animation Information---------------+\n");
  // Per ANIMATION DEBUG
  tIterator<Animation>* animIt = this->animationList->getIterator();
  Animation* anim = animIt->firstElement();
  while( anim != NULL)
    {
      //      anim->debug();
      anim = animIt->nextElement();
    }
  delete animIt;

  PRINT(0)("+--------------------------------AP--+\n");
}
