/*!
 * @file shell.h
 * Definition of a on-screen-shell
*/

#ifndef _SHELL_H
#define _SHELL_H

#include "element_2d.h"
#include "event_listener.h"

#include <stdarg.h>

// FORWARD DECLARATION
class Text;
template<class T> class tList;

//! A class that is able to redirect all output to a openGL-Shell, and that one can use to input some commands
/**
 * the major idea is, that all the Output can be redirected to the Shell,
 * and does not have to be displayed to the opening Shell, this is good,
 * for developers using Windows, where all output is otherwise redirected
 * to stdout.txt
 *
 * Furthermore the Shell should enable us, to input some simple commands
 * Each Class can tell check itself in to the Shell, and listen for commands.
 *
 * @todo implement what is written above :/
 */
class Shell : public Element2D, public EventListener {

  public:
    virtual ~Shell();
    /** @returns a Pointer to the only object of this Class */
    inline static Shell* getInstance() { if (!Shell::singletonRef) Shell::singletonRef = new Shell();  return Shell::singletonRef; };


    void setBufferSize(unsigned int bufferSize) { this->bufferSize = bufferSize; };

    void setBufferDisplaySize(unsigned int bufferDisplaySize);

    // BUFFER //
    void flushBuffers();
    static bool addBufferLineStatic(const char* line, ...);
    void addBufferLine(const char* line, va_list arg);
    void moveBuffer(int lineCount);
    const char* getBufferLine(unsigned int lineNumber);

    // InputLine
    void flushInputLine();
    void addCharacter(char character);
    void addCharacters(const char* characters);
    void removeCharacters(unsigned int characterCount = 1);
    bool executeCommand();


    // EventListener
    virtual void process(const Event &event);

    // Element2D-functions
    void tick(float dt);
    virtual void draw() const;

    void debug() const;

  private:
    bool autoComplete();


  private:
    Shell();
    static Shell*          singletonRef;           //!< The singleton-reference to the only memeber of this class.

    unsigned int           bufferSize;             //!< The Size of the buffer
    unsigned int           bufferDisplaySize;      //!< The Size of the Display-buffer, in lines (not in characters)

    Text*                  inputLineText;          //!< The inputLine of the Shell
    char*                  inputLine;              //!< the Char-Array of the Buffer
    float                  repeatDelay;            //!< The Repeat-Delay.
    float                  delayed;                //!< how much of the delay is remaining.
    int                    pressedKey;             //!< the pressed key that will be repeated.

    tList<char>*           buffer;                 //!< A list of stored char-arrays(strings) to store the history

    Text**                 bufferText;             //!< A list of stored bufferTexts for the display of the buffer
    unsigned int           textSize;               //!< The size of the text.
    unsigned int           lineSpacing;            //!< The Spacing between lines.

    char                   bufferArray[10000];     //!< a BUFFER for fast writing
};

#endif /* _SHELL_H */
