/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "element_2d.h"
#include "render_2d.h"

#include "graphics_engine.h"
#include "p_node.h"
#include "load_param.h"
#include "tinyxml.h"
#include "class_list.h"
#include "list.h"

using namespace std;

Element2D::Element2D()
{
  this->init();
  this->setParent2D(NullElement2D::getInstance());
}

/**
 * standard constructor
 * @todo this constructor is not jet implemented - do it
*/
Element2D::Element2D (Element2D* parent)
{
  this->init();

  if (this->parent != NULL)
    this->setParent2D(parent);
}

/**
 * standard deconstructor
*/
Element2D::~Element2D ()
{
  // delete what has to be deleted here
  Render2D::getInstance()->unregisterElement2D(this);

  tIterator<Element2D>* iterator = this->children->getIterator();
  Element2D* pn = iterator->nextElement();
  while( pn != NULL)
  {
    delete pn;
    pn = iterator->nextElement();
  }
  delete iterator;
  /* this deletes all children in the list */
  delete this->children;
  if (this->parent)
    this->parent->removeChild2D(this);

  if (this->toCoordinate != NULL)
    delete this->toCoordinate;
  if (this->toDirection != NULL)
    delete this->toDirection;
}


/**
 * initializes a Element2D
 */
void Element2D::init()
{
  this->setClassID(CL_ELEMENT_2D, "Element2D");

  this->setVisibility(true);
  this->setActiveness(true);
  this->setAlignment(E2D_ALIGN_NONE);
  this->layer = E2D_TOP;
  this->bindNode = NULL;

  this->setParentMode2D(E2D_PARENT_ALL);
  this->parent = NULL;
  this->children = new tList<Element2D>;
  this->relDirection = 0.0;
  this->bRelCoorChanged = true;
  this->bRelDirChanged = true;
  this->toCoordinate = NULL;
  this->toDirection = NULL;
//  else
  //  this->setParent2D(parent);

  Render2D::getInstance()->registerElement2D(this);
}

/**
 * Loads the Parameters of an Element2D from...
 * @param root The XML-element to load from
 */
void Element2D::loadParams(const TiXmlElement* root)
{
  LoadParam<Element2D>(root, "alignment", this, &Element2D::setAlignment)
      .describe("loads the alignment: (either: center, left, right or screen-center)");

  LoadParam<Element2D>(root, "layer", this, &Element2D::setLayer)
      .describe("loads the layer onto which to project: (either: top, medium, bottom, below-all)");

  LoadParam<Element2D>(root, "bind-node", this, &Element2D::setBindNode)
      .describe("sets a node, this 2D-Element should be shown upon (name of the node)");

  LoadParam<Element2D>(root, "visibility", this, &Element2D::setVisibility)
      .describe("if the Element is visible or not");

//   LoadParam<Element2D>(root, "2d-position", this, &Element2D::setPosition2D)
//       .describe("the _relative_ position (away from alignment) this 2d-element shows");

}

/**
 * sets the alignment of the 2D-element in form of a String
 * @param alignment the alignment @see loadParams
*/
void Element2D::setAlignment(const char* alignment)
{
  if (!strcmp(alignment, "center"))
    this->setAlignment(E2D_ALIGN_CENTER);
  else if (!strcmp(alignment, "left"))
    this->setAlignment(E2D_ALIGN_LEFT);
  else if (!strcmp(alignment, "right"))
    this->setAlignment(E2D_ALIGN_RIGHT);
  else if (!strcmp(alignment, "screen-center"))
    this->setAlignment(E2D_ALIGN_SCREEN_CENTER);
}


/**
 * moves a Element to another layer
 * @param layer the Layer this is drawn on
 */
void Element2D::setLayer(E2D_LAYER layer)
{
  Render2D::getInstance()->moveToLayer(this, layer);
  this->layer = layer;
}

/**
 * sets the layer onto which this 2D-element is projected to.
 * @param layer the layer @see loadParams
 */
void Element2D::setLayer(const char* layer)
{
  if (!strcmp(layer, "top"))
    this->setLayer(E2D_TOP);
  else if (!strcmp(layer, "medium"))
    this->setLayer(E2D_MEDIUM);
  else if (!strcmp(layer, "bottom"))
    this->setLayer(E2D_BOTTOM);
  else if (!strcmp(layer, "below-all"))
    this->setLayer(E2D_BELOW_ALL);
}


/**
 * sets a node, this 2D-Element should be shown upon
 * @param bindNode the name of the Node (should be existing)
 */
void Element2D::setBindNode(const char* bindNode)
{
  const PNode* tmpBindNode = dynamic_cast<const PNode*>(ClassList::getObject(bindNode, CL_PARENT_NODE));
  if (tmpBindNode != NULL)
    this->bindNode = tmpBindNode;
}


void Element2D::setRelCoor2D (const Vector& relCoord)
{
  this->relCoordinate = relCoord;
  this->bRelCoorChanged = true;
}


void Element2D::setRelCoor2D (float x, float y, float z)
{
  this->setAbsCoor2D(Vector(x,y,z));
}

void Element2D::setRelCoor2Dpx (int x, int y)
{
  this->setRelCoor2D(Vector((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                     (float)y/(float)GraphicsEngine::getInstance()->getResolutionY(),
                     0
                           ));
}


void Element2D::setRelCoorSoft2D(const Vector& relCoordSoft, float bias)
{
  if (likely(this->toCoordinate == NULL))
    this->toCoordinate = new Vector();

  *this->toCoordinate = relCoordSoft;
  this->bias = bias;
}

void Element2D::setRelCoorSoft2Dpx (int x, int y, float bias)
{
  this->setRelCoorSoft2D(Vector((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                         (float)y/(float)GraphicsEngine::getInstance()->getResolutionY(),
                         0),
                         bias);
}


void Element2D::setRelCoorSoft2D(float x, float y, float depth, float bias)
{
  this->setRelCoorSoft2D(Vector(x, y, depth), bias);
}

void Element2D::setAbsCoor2D (const Vector& absCoord)
{
  if( likely(this->parentMode & E2D_PARENT_MOVEMENT))
  {
    /* if you have set the absolute coordinates this overrides all other changes */
    if (likely(this->parent != NULL))
      this->relCoordinate = absCoord - parent->getAbsCoor2D ();
    else
      this->relCoordinate = absCoord;
  }
  if( this->parentMode & E2D_PARENT_ROTATE_MOVEMENT)
  {
    if (likely(this->parent != NULL))
      this->relCoordinate = absCoord - parent->getAbsCoor2D ();
    else
      this->relCoordinate = absCoord;
  }

  this->bRelCoorChanged = true;
}

void Element2D::setAbsCoor2D (float x, float y, float depth)
{
  this->setAbsCoor2D(Vector(x, y, depth));
}


void Element2D::setAbsCoor2Dpx (int x, int y)
{
  this->setAbsCoor2D(Vector((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                     (float)y/(float)GraphicsEngine::getInstance()->getResolutionY(),
                     0
                           ));
}

void Element2D::shiftCoor2D (const Vector& shift)
{
  this->relCoordinate += shift;
  this->bRelCoorChanged = true;

}


void Element2D::shiftCoor2Dpx (int x, int y)
{
  this->shiftCoor2D(Vector((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                    (float)y/(float)GraphicsEngine::getInstance()->getResolutionY(),
                     0));
}


void Element2D::setRelDir2D (float relDir)
{
  this->relDirection = relDir;
  this->bRelDirChanged = true;
}

void Element2D::setRelDirSoft2D(float relDirSoft, float bias)
{
  if (likely(this->toDirection == NULL))
    this->toDirection = new float;

  *this->toDirection = relDirSoft;
  this->bias = bias;
}

void Element2D::setAbsDir2D (float absDir)
{
  if (likely(this->parent != NULL))
    this->relDirection = absDir - this->parent->getAbsDir2D();
  else
    this->relDirection = absDir;

  this->bRelDirChanged = true;
}

void Element2D::shiftDir2D (float shiftDir)
{
  this->relDirection = this->relDirection + shiftDir;
  this->bRelDirChanged = true;
}


void Element2D::addChild2D (Element2D* child, int parentingMod)
{
  if( likely(child->parent != NULL))
  {
    PRINTF(4)("Element2D::addChild() - reparenting node: removing it and adding it again\n");
    child->parent->children->remove(child);
  }
  child->parentMode = parentMode;
  child->parent = this;
  this->children->add(child);
  child->parentCoorChanged();
}

void Element2D::addChild2D (const char* childName)
{
  Element2D* childNode = dynamic_cast<Element2D*>(ClassList::getObject(childName, CL_ELEMENT_2D));
  if (childNode != NULL)
    this->addChild2D(childNode);
}

void Element2D::removeChild2D (Element2D* child)
{
  child->remove2D();
  this->children->remove(child);
  child->parent = NULL;
}

void Element2D::remove2D()
{
  tIterator<Element2D>* iterator = this->children->getIterator();
  Element2D* pn = iterator->nextElement();

  while( pn != NULL)
  {
    NullElement2D::getInstance()->addChild2D(pn, pn->getParentMode2D());
    pn = iterator->nextElement();
  }
  delete iterator;
  this->parent->children->remove(this);
}


void Element2D::setParent2D (Element2D* parent)
{
  parent->addChild2D(this);
}

void Element2D::setParent2D (const char* parentName)
{
  Element2D* parentNode = dynamic_cast<Element2D*>(ClassList::getObject(parentName, CL_ELEMENT_2D));
  if (parentNode != NULL)
    parentNode->addChild2D(this);

}


void Element2D::softReparent2D(Element2D* parentNode, float bias)
{
  if (this->parent == parentNode)
    return;

  if (likely(this->toCoordinate == NULL))
  {
    this->toCoordinate = new Vector();
    *this->toCoordinate = this->getRelCoor2D();
  }
  if (likely(this->toDirection == NULL))
  {
    this->toDirection = new float;
    *this->toDirection = this->getRelDir2D();
  }
  this->bias = bias;


  Vector tmpV = this->getAbsCoor2D();
  float tmpQ = this->getAbsDir2D();

  parentNode->addChild2D(this);

  if (this->parentMode & PNODE_ROTATE_MOVEMENT)
    ;//this->setRelCoor(this->parent->getAbsDir().inverse().apply(tmpV - this->parent->getAbsCoor()));
  else
    this->setRelCoor2D(tmpV - parentNode->getAbsCoor2D());

  this->setRelDir2D(tmpQ - parentNode->getAbsDir2D());
}

void Element2D::softReparent2D(const char* parentName, float bias)
{
  Element2D* parentNode = dynamic_cast<Element2D*>(ClassList::getObject(parentName, CL_ELEMENT_2D));
  if (parentNode != NULL)
    this->softReparent2D(parentNode, bias);
}


void Element2D::setParentMode2D (const char* parentingMode)
{
  this->setParentMode2D(Element2D::charToParentingMode2D(parentingMode));
}


void Element2D::update2D (float dt)
{
  // setting the Position of this 2D-Element.
  if( likely(this->parent != NULL))
  {
      // movement for nodes with smoothMove enabled
    if (unlikely(this->toCoordinate != NULL))
    {
      Vector moveVect = (*this->toCoordinate - this->getRelCoor2D()) *dt*bias;

      if (likely(moveVect.len() >= .001))//PNODE_ITERATION_DELTA))
      {
        this->shiftCoor2D(moveVect);
      }
      else
      {
        delete this->toCoordinate;
        this->toCoordinate = NULL;
        PRINTF(5)("SmoothMove of %s finished\n", this->getName());
      }
    }
    if (unlikely(this->toDirection != NULL))
    {
      float rotFlot = (*this->toDirection - this->getRelDir2D()) *dt*bias;
      if (likely(rotFlot >= .001))//PNODE_ITERATION_DELTA))
      {
        this->shiftDir2D(rotFlot);
      }
      else
      {
        delete this->toDirection;
        this->toDirection = NULL;
        PRINTF(5)("SmoothRotate of %s finished\n", this->getName());
      }
    }

    // MAIN UPDATE /////////////////////////////////////
    this->lastAbsCoordinate = this->absCoordinate;

    PRINTF(5)("Element2D::update - %s - (%f, %f, %f)\n", this->getName(), this->absCoordinate.x, this->absCoordinate.y, this->absCoordinate.z);


    if( this->parentMode & PNODE_LOCAL_ROTATE && this->bRelDirChanged)
    {
      /* update the current absDirection - remember * means rotation around sth.*/
      this->prevRelDirection = this->relDirection;
      this->absDirection = this->relDirection + parent->getAbsDir2D();;
    }


    if (this->alignment == E2D_ALIGN_SCREEN_CENTER && this->bRelCoorChanged)
    {
      this->prevRelCoordinate = this->relCoordinate;
      this->absCoordinate.x = .5 + this->relCoordinate.x;
      this->absCoordinate.y = .5 + this->relCoordinate.y;
      this->absCoordinate.z = 0.0;
    }
    else if (this->bindNode)
    {
      GLdouble projectPos[3];
      gluProject(this->bindNode->getAbsCoor().x,
                 this->bindNode->getAbsCoor().y,
                 this->bindNode->getAbsCoor().z,
                 GraphicsEngine::modMat,
                 GraphicsEngine::projMat,
                 GraphicsEngine::viewPort,
                 projectPos,
                 projectPos+1,
                 projectPos+2);
      this->absCoordinate.x = projectPos[0]/(float)GraphicsEngine::getInstance()->getResolutionX() + this->relCoordinate.x;
      this->absCoordinate.y = projectPos[1]/(float)GraphicsEngine::getInstance()->getResolutionY() + this->relCoordinate.y;
      this->absCoordinate.z = projectPos[2] + this->relCoordinate.z;
    }
    else
    {
      if(likely(this->parentMode & PNODE_MOVEMENT && this->bRelCoorChanged))
      {
        /* update the current absCoordinate */
        this->prevRelCoordinate = this->relCoordinate;
        this->absCoordinate = this->parent->getAbsCoor2D() + this->relCoordinate;
      }
      else if( this->parentMode & PNODE_ROTATE_MOVEMENT && this->bRelCoorChanged)
      {
        /* update the current absCoordinate */
        this->prevRelCoordinate = this->relCoordinate;
        float sine = sin(this->parent->getAbsDir2D());
        float cose = cos(this->parent->getAbsDir2D());
//        this->absCoordinate.x = this->relCoordinate.x*cose - this->relCoordinate.y*sine + this->parent->getRelCoor2D().x*(1-cose) +this->parent->getRelCoor2D().y*sine;
//        this->absCoordinate.y = this->relCoordinate.x*sine + this->relCoordinate.y*cose + this->parent->getRelCoor2D().y*(1-cose) +this->parent->getRelCoor2D().x*sine;

        this->absCoordinate.x = this->parent->getAbsCoor2D().x + (this->relCoordinate.x*cos(this->parent->getAbsDir2D()) - this->relCoordinate.y * sin(this->parent->getAbsDir2D()));
        this->absCoordinate.y = this->parent->getAbsCoor2D().y + (this->relCoordinate.x*sin(this->parent->getAbsDir2D()) + this->relCoordinate.y * cos(this->parent->getAbsDir2D()));

      }
    }
    /////////////////////////////////////////////////
  }
  else
  {
    PRINTF(5)("Element2D::update - (%f, %f, %f)\n", this->absCoordinate.x, this->absCoordinate.y, this->absCoordinate.z);
    if (this->bRelCoorChanged)
      this->absCoordinate = this->relCoordinate;
    if (this->bRelDirChanged)
      this->absDirection = this->getAbsDir2D() * this->relDirection;
  }


  // UPDATE CHILDREN
  if(this->children->getSize() > 0)
  {
    tIterator<Element2D>* iterator = this->children->getIterator();
    Element2D* pn = iterator->nextElement();
    while( pn != NULL)
    {
      /* if this node has changed, make sure, that all children are updated also */
      if( likely(this->bRelCoorChanged))
        pn->parentCoorChanged ();
      if( likely(this->bRelDirChanged))
        pn->parentDirChanged ();

      pn->update2D(dt);
          //pn = this->children->nextElement();
      pn = iterator->nextElement();
    }
    delete iterator;
  }

  // FINISHING PROCESS
  this->velocity = (this->absCoordinate - this->lastAbsCoordinate) / dt;
  this->bRelCoorChanged = false;
  this->bRelDirChanged = false;
}


void Element2D::debug (unsigned int depth, unsigned int level) const
{
  for (unsigned int i = 0; i < level; i++)
    PRINT(0)(" |");
  if (this->children->getSize() > 0)
    PRINT(0)(" +");
  else
    PRINT(0)(" -");
  PRINT(0)("Element2D(%s::%s) - absCoord: (%0.2f, %0.2f), relCoord(%0.2f, %0.2f), direction(%0.2f) - %s\n",
  this->getClassName(),
  this->getName(),
  this->absCoordinate.x,
  this->absCoordinate.y,
  this->relCoordinate.x,
  this->relCoordinate.y,
  this->getAbsDir2D(),
  Element2D::parentingModeToChar2D(parentMode));
  if (depth >= 2 || depth == 0)
  {
    tIterator<Element2D>* iterator = this->children->getIterator();
      //PNode* pn = this->children->enumerate ();
    Element2D* pn = iterator->nextElement();
    while( pn != NULL)
    {
      if (depth == 0)
        pn->debug(0, level + 1);
      else
        pn->debug(depth - 1, level +1);
      pn = iterator->nextElement();
    }
    delete iterator;
  }
}

#include "color.h"

void Element2D::debugDraw2D(unsigned int depth, float size, Vector color) const
{

}


// helper functions //
const char* Element2D::parentingModeToChar2D(int parentingMode)
{
  if (parentingMode == E2D_PARENT_LOCAL_ROTATE)
    return "local-rotate";
  else if (parentingMode == E2D_PARENT_ROTATE_MOVEMENT)
    return "rotate-movement";
  else if (parentingMode == E2D_PARENT_MOVEMENT)
    return "movement";
  else if (parentingMode == E2D_PARENT_ALL)
    return "all";
  else if (parentingMode == E2D_PARENT_ROTATE_AND_MOVE)
    return "rotate-and-move";
}

E2D_PARENT_MODE Element2D::charToParentingMode2D(const char* parentingMode)
{
  if (!strcmp(parentingMode, "local-rotate"))
    return (E2D_PARENT_LOCAL_ROTATE);
  else  if (!strcmp(parentingMode, "rotate-movement"))
    return (E2D_PARENT_ROTATE_MOVEMENT);
  else  if (!strcmp(parentingMode, "movement"))
    return (E2D_PARENT_MOVEMENT);
  else  if (!strcmp(parentingMode, "all"))
    return (E2D_PARENT_ALL);
  else  if (!strcmp(parentingMode, "rotate-and-move"))
    return (E2D_PARENT_ROTATE_AND_MOVE);
}






/**
 * ticks the 2d-Element
 * @param dt the time elapsed since the last tick
 */
void Element2D::tick(float dt)
{

}







NullElement2D* NullElement2D::singletonRef = 0;

/**
 *  creates the one and only NullElement2D
 * @param absCoordinate the cordinate of the Parent (normally Vector(0,0,0))
 */
NullElement2D::NullElement2D () : Element2D(NULL)
{
  this->setClassID(CL_NULL_PARENT, "NullElement2D");
  this->setName("NullElement2D");

  this->setParentMode2D(E2D_PARENT_ALL);
  NullElement2D::singletonRef = this;
}


/**
 *  standard deconstructor
 */
NullElement2D::~NullElement2D ()
{
  //delete singletonRef;
  NullElement2D::singletonRef = NULL;
}
