/*!
 \file gui_gtk.h
 \brief Contains all th different Widgets.
*/
#ifndef _GUI_GTK_H
#define _GUI_GTK_H

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "debug.h"
#include "globals.h"

#ifdef HAVE_GTK2
#include <gtk/gtkmain.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkhscale.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkeventbox.h>
#include <gtk/gtkprogressbar.h>
#include <gtk/gtkfilesel.h>
#endif /* HAVE_GTK2 */

// enumerator for different GuiOption-Types
enum GUI_OPTION {GUI_BOX = -2,
                 GUI_CONTAINER = -1,
                 GUI_NOTHING = 0,
                 GUI_BOOL = 1,
                 GUI_INT = 2,
                 GUI_FLOAT = 3,
                 GUI_CHAR = 4,
                 GUI_CHAR_ARRAY = 5};

extern char* executable;

bool initGUI(int argc, char* argv[]);
bool mainloopGUI();

//! This is the topmost object that can be displayed all others are derived from it.
class Widget
{
 private:

 public:
  Widget();
  virtual ~Widget();

  void show();
  void hide();
  void setSize(int width, int height);

  virtual void setTitle(const char* title);  //!< An abstract Function, that sets the title of Widgets.
  virtual const char* getTitle() const { return this->title; };

  Widget* findWidgetByName(char* name, unsigned int depth);
  void walkThrough(void(*function)(Widget*), unsigned int depth);
  void walkThrough(void(*function)(Widget*, void*), void* data, unsigned int depth);
   static void listOptionsAndGroups(Widget* widget);
   static void listOptions(Widget* widget);
   static void listOptions(Widget* widget, void* data);
   Widget* findOptionByNumber(int* number, unsigned int depth);
   static void listGroups(Widget* widget);
   static void listGroups(Widget* widget, void* data);
   static void printHelp(Widget* widget);
   Widget* findGroupByNumber(int* number, unsigned int depth);
   static void setOptions(Widget* widget);
   static void redrawOptions(Widget* widget);
   static void flagCheck(Widget* widget, void* flagName);

#ifdef HAVE_GTK2
  // Connection - Functions
  gulong connectSignal(char* event, gint(*signal)(GtkWidget*, GdkEvent*, void* ));
  gulong connectSignal(char* event, gint(*signal)(GtkWidget*, Widget* ));
  gulong connectSignal(char* event, void* extObj, gint(*signal)(GtkWidget*, GdkEvent*, void* ));
  gulong connectSignal(char* event, void* extObj, gint(*signal)(GtkWidget*, void* ));
  gulong connectSignal(char* event, void* extObj, gint(*signal)(GtkWidget*, GdkEventKey*, void* ));
  void disconnectSignal(gulong signalID);
  // Signals
  static gint doNothingSignal(GtkWidget* widget, GdkEvent* event, void* nothing);
#else /* HAVE_GTK2 */
  // Connection - Functions
  unsigned long connectSignal(char* event, int(*signal)(void*, void*, void* )){};
  unsigned long connectSignal(char* event, int(*signal)(void*, Widget* )){};
  unsigned long connectSignal(char* event, void* extObj, int(*signal)(void*, void*, void* )){};
  unsigned long connectSignal(char* event, void* extObj, int(*signal)(void*, void* )){};
  void disconnectSignal(unsigned long signalID);
  // Signals
  static int doNothingSignal(void* widget, void* event, void* nothing);
#endif /* HAVE_GTK2 */


  Widget* next;                     //!< next always points to the next Widget in the list. Every Widget has a next one, or has NULL as next
#ifdef HAVE_GTK2
  GtkWidget* widget;                //!< widget is the gtk_widget that the specific Object Contains.
#else /* HAVE_GTK2 */
  void* widget;
#endif /* HAVE_GTK2 */

  GUI_OPTION optionType;            //!< The Type of the Widget.
  char* title;                      //!< The name of the Widget. Some do display it, Options need it to save;
};

//! This is a Packer Object, which has the ability to Pack other Widgets into itself.
class Packer : public Widget
{
 public:
  Packer();
  virtual ~Packer();

  Widget* down;                      //!< this points to the Widget below this.
  char* groupName;                   //!< For each Packer you can specify a Groupname under which the lowerWidgets will be saved.

  void setGroupName(const char* name);
  /** @returns the GroupName if existent NULL otherwise */
  inline const char* getGroupName() const {return this->groupName;}


  virtual void fill(Widget* lowerWidget) = 0; //!< An abstract function, that fills Packers.
};

//! This is a Container Class, it can contain one sub-Widget: down.
/**
 * A Container is a Widget that can hold a subWidget in addition to a next-Widget.
 * The Container can by itself not be displayed created or used.
 * The derived classes of Container can be displayed
*/
class Container : public Packer
{
 private:
  int borderwidth;          //!< The width of The Container Boarder.
  int policy;               //!< The Update Policy of a Container.

 public:
  Container();
  virtual ~Container();

  void setBorderWidth(int borderwidth);
  void fill(Widget* lowerWidget);
};

//! Window is the class that creates new Windows, and handels them
/**
 * A Window is a class derived from Container that contains a window-widget.
 * It has the ability to hold one sub-object
 */
class Window : public Container
{
 private:
  bool isOpen;                      //!< A bool Variable that checks if a Window is already open.
 public:
  static Window* mainWindow;        //!< Pointer to the First Window that was opened. By default this should be the GUI's main-Window.
  static void addWindow(Window* windowToAdd);

  Window(const char* windowName = NULL);
  virtual ~Window();

  virtual void setTitle(const char* title);
  void showall();
  void open();
  void close();

#ifdef HAVE_GTK2
  // Signals
  static gint windowOpen(GtkWidget* widget, GdkEvent* event, void* window);
  static gint windowClose(GtkWidget* widget, GdkEvent* event, void* window);
#else /* HAVE_GTK2 */
  int Window::windowOpen(void* widget, void* event, void* window);
  int Window::windowClose(void* widget, void* event, void* window);
#endif /* HAVE_GTK2 */
};

//! Frame is the class that handles frames
/**
 * A Frame is an object, that has a border and if you like a name on it.
 * It can contain a Widget, which means that you can insert anything you like inside of a frame
 */
class Frame :public Container
{
 public:
  Frame(const char* frameName = NULL);
  virtual ~Frame();

  virtual void setTitle(const char* title);
};

//! EventBox is a Container that can Handle all Events happening inside of it.
/**
 * Example: if you have a picture, and you want it to catch mouse-clicks, you have to pack it inside a EventBox
 */
class EventBox : public Container
{
 public:
  EventBox(const char* eventBoxName = NULL);
  virtual ~EventBox();
};

//! A Box can contain multiple Widgets
/**
   A Box can Contain multiple Widgets, that are ordered either horizontally or vertically
   I defined the standartbox to be horizontally.
   A Box is always filled left->right(horizontally) or up->down(vertically)
*/
class Box : public Packer
{
 public:
  Box(char boxtype = 'h');
  virtual ~Box();

  virtual void fill(Widget* lowerWidget);
};

//! An Option is a Widget that contains something that may change its state.
/**
 * Options are the essence of a GUI, they: Configure, Start, Quit, Execute, and make it worth something
 */
class Option : public Widget
{
 protected:
  bool saveable;              //!< Options can be Saved.

 public:
  Option();
  virtual ~Option();

  int value;                  //!< every option has a value either true or false(0,1) or something else like 25 for 25% of the volume
  char* flagName;             //!< options have a flag name that will be appendet if you start the Program from the GUI.
  char* flagNameShort;        //!< like flag_name but shorter
  int defaultValue;           //!< A default value is good, for hiding a option if it is not needed.(hidden if value == default_value)

  char* shortDescription;      //!< A Text that describes this option in short
  char* longDescription;      //!< A Longer Text describing this option in a full way

  void saveability(bool isSaveable = true);
  virtual char* save();
  virtual void load(const char* loadString);

  bool isSaveable();
  void setDefaultValue(int defaultValue);
  void setFlagName(const char* flagname, int defaultvalue);
  void setFlagName(const char* flagname, const char* flagnameshort, int defaultvalue);
  void setDescription(const char* shortDescription, const char* longDescription = NULL);

  virtual void redraw() = 0;       //!< A Option must be able to redraw itself.
  virtual void changeOption() = 0; //!< What to do, if an Option is Changed. eacht option decides for itself.
#ifdef HAVE_GTK2
    // Signals
  static gint OptionChange(GtkWidget* widget, Widget* option); //!< Signal for Options that change.
#endif /* HAVE_GTK2 */
};

//! Buttons can be pressed, and released.
/**
 * Buttons are mainly there for executing some action like Starting the Programm, or Quiting it.
 */
class Button : public Option
{
 public:
  Button(const char* buttonName = NULL);
  virtual ~Button();

  virtual void setTitle(const char* title);
  virtual void redraw();
  virtual void changeOption();
};

//! CheckButtons are a key in configuring bool Variables
/** CheckButtons can configure bool Variables like wireframe on/off, enable_sound etc.
 */
class CheckButton : public Option
{
 public:
  CheckButton(const char* buttonName = NULL);
  virtual ~CheckButton();

  bool isActive();

  virtual void setTitle(const char* title);
  virtual void redraw();
  virtual void changeOption();
};

//! Sliders are Options that can be modified in their value
/**
 * good for volume, brightness, etc.
 */
class Slider : public Option
{
 private:
  float start;                            //!< The beginning of the Slider-range.
  float end;                              //!< The end of the Slider-range.
  float fValue;                           //!< a value for the slider
 public:
  Slider(const char* slidername, float start, float end);
  virtual ~Slider();

  void setExactness(int exactness);
  void setValue(float value);
  virtual void redraw();
  virtual void changeOption();

  virtual char* save();
  virtual void load(const char* loadString);
};

//! A Menu is an Option that has a dropdown menu, where you can chose between different Items
class Menu : public Option
{
 private:
#ifdef HAVE_GTK2
  GtkWidget* menu;                      //!< The menu That will hold the Options.
#endif /* HAVE_GTK2 */

  //! A struct to handle the MenuItems
  struct MenuItem
  {
    char* name;                         //!< The name of this entry.
    int itemNumber;                     //!< The n'th entry of this menu;
#ifdef HAVE_GTK2
    GtkWidget* item;                    //!< One Item From a Menu.
#endif /* HAVE_GTK2 */

    MenuItem* next;                     //!< Pointer to the next MenuItem.
  };
  MenuItem* firstItem;                  //!< Pointer to the first Item.
  MenuItem* currItem;                   //!< Pointer to the current Item.

 public:
  Menu(const char* menuName);
  Menu(char* menuname, ...);
  virtual ~Menu();
  void init();

  virtual char* save();
  virtual void load(const char* loadString);

  void addItem(char* itemName);
  virtual void redraw();
  virtual void changeOption();
};

//! A OptionLabel is a simple Label, that holds a char*, and will be updated, if changed.
class OptionLabel : public Option
{
 public:
  OptionLabel(const char* label, const char* value);
  virtual ~OptionLabel();

  void setValue(const char* newValue);

  virtual char* save();
  virtual void load(const char* loadString);

  virtual void redraw();
  virtual void changeOption();

  char* cValue;                          //!< The Value the Label will have. @todo make private
};

//! A EntryField is a TextEntry field, for putting some text into.
class EntryField : public Option
{
 public:
  EntryField(const char* name = NULL);
  virtual ~EntryField();

  void setValue(const char* newValue);
  virtual char* save();
  virtual void load(const char* loadString);

  virtual void redraw();
  virtual void changeOption();
};

//! A label is a Widget, that displays a text
class Label : public Widget
{
 public:
  Label(const char* text = NULL);
  virtual ~Label();

  virtual void setTitle(const char* text);
  void ereaseText();
  void appendText(char* textToAppend);
  void appendInt(int intToAppend);
  const char* getText();
};

//! A ProgressBar is a Widget, that can display a Progress
class ProgressBar : public Widget
{
 public:
  ProgressBar(const char* label = NULL);
  virtual ~ProgressBar();

  void setProgress(double progress);
  void setTotalSize(double totalSize);
  double getProgress();

 private:
  double totalSize;         //!< The total Size of a download Bar
  double progress;          //!< The progress of a Bar.
#ifdef HAVE_GTK2
  GtkAdjustment* adjustment;
#endif /* HAVE_GTK2 */
};

//! Image is the keeper of one Image
/**
 * Images are mighty cool.
 * Images can help you lighten up the Programming process, and will give everyone a better impression of the Software.
 */
class Image : public Widget
{
 public:
  Image(const char* imgaeName);
  Image(char** imageData);
  virtual ~Image();
  void init(const char* name);
};

//! A FileDialog is a window with wich one can select a File
class FileDialog : public Widget
{
 private:
  OptionLabel* changeOption;
  Button* openUpButton;
  bool isOpen;
  bool (*okFunc)(const char* , void*);
  void* okObject;

 public:
  FileDialog(const char* fileDialogName);
  virtual ~FileDialog();

  void setChangeOption(OptionLabel* changeOption);
  void setOKFunc(void* okObject, bool(*function)(const char* , void*));
  void setOpenUpButton(Button* openUpButton);
  void setDefaultFileName(const char* defaultFileName);
  void setMask(const char* mask);
  void disableFileOpts();

  void okEvent();
  void open();
  void close();

#ifdef HAVE_GTK2
  static gint dialogOK(GtkWidget* widget, GdkEvent* event, void* dialog);
  static gint dialogOpen(GtkWidget* widget, GdkEvent* event, void* dialog);
  static gint dialogClose(GtkWidget* widget, GdkEvent* event, void* dialog);
#else /* HAVE_GTK2 */
  static int dialogOK(void* widget, void* event, void* dialog);
  static int dialogOpen(void* widget, void* event, void* dialog);
  static int dialogClose(void* widget, void* event, void* dialog);
#endif /* HAVE_GTK2 */
};

#endif /* _GUI_GTK_H */
