/*!
    \file p_node.h
  *  Definition of a parenting node

    parenting is how coordinates are handled in orxonox, meaning, that all coordinates
    are representet relative to another parent node. this nodes build a parenting
    tree of one-sided references (from up to down referenced).
    Every node manages itself a list of childrens (of whos it is parent - easy...)

    absCoordinate, absDirection have to be recalculated as soon as there was a change in
    place or ortientation. this is only the case if
    o bDirChanged is true (so changed) AND timeStamp != now
    o bCoorChanged is true (so moved) AND timeStamp != now
    this conditions make it cheaper to recalculate the tree (reduces redundant work).

    remember: if you have to change the coordinates or the directions, use the functions
    that are defined to execute this operation - otherwhise there will be big problems...
*/


#ifndef _P_NODE_H
#define _P_NODE_H

#include "base_object.h"
#include "vector.h"

// FORWARD DEFINITION \\
class PNode; /* forward decleration, so that parentEntry has access to PNode */
//class Quaternion;
//class Vector;
class TiXmlElement;
template<class T> class tList;

//! Parental linkage modes
typedef enum
{
  PNODE_LOCAL_ROTATE          =   1,    //!< Rotates all the children around their centers.
  PNODE_ROTATE_MOVEMENT       =   2,    //!< Moves all the children around the center of their parent, without the rotation around their own centers.

  PNODE_MOVEMENT              =   4,    //!< Moves all children along with the parent.
// special linkage modes
  PNODE_ALL                   =   3,    //!< Moves all children around the center of their parent, and also rotates their centers
  PNODE_ROTATE_AND_MOVE       =   5    //!< Rotates all children around their axis, and moves them as the Parent Moves, but does not rotate around the center of their parent.

} PARENT_MODE;

//! The default mode of the translation-binding.
#define DEFAULT_MODE PNODE_ALL


//! Patent Node is a Engine to calculate the position of an Object in respect to the position of its parent.
class PNode : virtual public BaseObject {

 public:
  PNode ();
  PNode(const TiXmlElement* root);
  PNode (const Vector& absCoordinate, PNode* pNode);
  virtual ~PNode ();

  void loadParams(const TiXmlElement* root);

  void setRelCoor (const Vector& relCoord);
  void setRelCoor (float x, float y, float z);
  /** @returns the relative position */
  inline const Vector& getRelCoor () const { return this->relCoordinate; };
  void setAbsCoor (const Vector& absCoord);
  void setAbsCoor (float x, float y, float z);
  /** @returns the absolute position */
  inline const Vector& getAbsCoor () const { return this->absCoordinate; };
  void shiftCoor (const Vector& shift);

  void setRelDir (const Quaternion& relDir);
  void setRelDir (float x, float y, float z);
  /** @returns the relative Direction */
  inline const Quaternion& getRelDir () const { return this->relDirection; };
  /** @returns a Vector pointing into the relative Direction */
  inline Vector getRelDirV() const { return this->relDirection.apply(Vector(0,1,0)); };
  void setAbsDir (const Quaternion& absDir);
  void setAbsDir (float x, float y, float z);
  /** @returns the absolute Direction */
  inline const Quaternion& getAbsDir () const { return this->absDirection; };
  /** @returns a Vector pointing into the absolute Direction */
  inline Vector getAbsDirV() const { return this->absDirection.apply(Vector(0,1,0)); };
  void shiftDir (const Quaternion& shift);

  /** @returns the Speed of the Node */
  inline float getSpeed() const {return this->velocity.len();}
  /** @returns the Velocity of the Node */
  inline const Vector& getVelocity() const {return this->velocity;}


  void addChild (PNode* pNode, int parentingMode = DEFAULT_MODE);
  void addChild (const char* childName);
  void removeChild (PNode* pNode);
  void remove();

  void setParent (PNode* parent);
  void setParent (const char* name);
  /** @returns the parent of this PNode */
  PNode* getParent () const { return this->parent; };

  void setParentMode (PARENT_MODE parentMode);
  void setParentMode (const char* parentingMode);
  /** @returns the Parenting mode of this node */
  int getParentMode() const { return this->parentMode; };

  void update (float dt);

  void debug (unsigned int depth = 1, unsigned int level = 0) const;
  void debugDraw(float size = 1.0) const;

 private:
  void init(PNode* parent);
  /** \brief tells the child that the parent's Coordinate has changed */
  inline void parentCoorChanged () { this->bRelCoorChanged = true; }
  /** \brief tells the child that the parent's Direction has changed */
  inline void parentDirChanged () { this->bRelDirChanged = true; }
  /** @returns the last calculated coordinate */
  inline Vector getLastAbsCoor() {return this->lastAbsCoordinate;}


 private:
  bool            bAbsCoorChanged;    //!< If Absolute Coordinate has changed since last time we checked
  bool            bRelCoorChanged;    //!< If Relative Coordinate has changed since last time we checked
  bool            bAbsDirChanged;     //!< If Absolute Direction has changed since last time we checked
  bool            bRelDirChanged;     //!< If Relative Direction has changed since last time we checked

  Vector          relCoordinate;      //!< coordinates relative to the parent
  Vector          absCoordinate;      //!< absolute coordinates in the world ( from (0,0,0) )
  Quaternion      relDirection;       //!< direction relative to the parent
  Quaternion      absDirection;       //!< absolute direvtion in the world ( from (0,0,1) )

  Vector          velocity;           //!< Saves the velocity.
  Vector          lastAbsCoordinate;  //!< this is used for speedcalculation, it stores the last coordinate

  PNode*          parent;             //!< a pointer to the parent node
  tList<PNode>*   children;           //!< list of the children

  unsigned int    parentMode;         //!< the mode of the binding
};

#endif /* _P_NODE_H */

