/*!
    \file weapon.h
  *  a weapon that a  can use


    A weapon is characterized by:
     o firing-rate: the initial firing rate of a weapon (1/s = Herz)
     o slowdown-factor: this is a factor d: exp(-d*x), d is element of all positive R. it determines how fast the firing-rate will slow down. if no slowdown: d=0, the bigger d is, the faster the weapon will slow down!
     o energy-consumption: this determines the energy that has to be used to produce this projectile = costs per projectile

    Furthermore there are some other attributes, that will help to represent a firing
    weapon in this world:
     o sound file/ressource: this is a pointer to the sound-file/ressource. however it may be represented
     o animations
*/


#ifndef _WEAPON_H
#define _WEAPON_H

#include "base_object.h"
#include "world_entity.h"

// FORWARD DECLARATION
class Projectile;
class Weapon;
class Animation3D;
class TiXmlElement;

//! An enumerator defining actions a Weapon can take
typedef enum {
  WA_NONE          =    0,    //!< No Action taken
  WA_SHOOT         =    1,    //!< emitting Shot
  WA_CHARGE        =    2,    //!< charge action (one click before the shot)
  WA_RELOAD        =    3,    //!< reload right after shoot is finished
  WA_ACTIVATE      =    4,    //!< activate the GUN
  WA_DEACTIVATE    =    5,    //!< deactivate the GUN
  WA_SPECIAL1      =    6,    //!< Special Action taken

  WA_ACTION_COUNT  =    7     //!< This must match the count of enumerations-members.
} WeaponAction;

//! An enumerator defining the States of a Weapon
typedef enum {
  WS_NONE          =    0,    //!< No State at all (if set, there is something wrong, or the weapon is not yet availiable)
  WS_SHOOTING      =    1,    //!< The State of the Shooting
  WS_CHARGING      =    2,    //!< The state of charging th weapon
  WS_RELOADING     =    3,    //!< The State of the Reloading
  WS_ACTIVATING    =    4,    //!< The State in which the weapon gets activated
  WS_DEACTIVATING  =    5,    //!< The State in which the weapon gets deactivated
  WS_INACTIVE      =    6,    //!< The State where the weapon is inactive (unable to shoot)
  WS_IDLE          =    7,    //!< The State where the weapon is idle

  WS_STATE_COUNT  =     8     //!< This must match the count of enumerations-members.
} WeaponState;


//! a weapon can be left or right sided
/**
 * @todo this will be reset with mirror X/Y/Z
 */
#define    W_LEFT        0
#define    W_RIGHT       1

//! An abstract class, that describes weapons
/**
 * This is used as a container for all the different kinds of weapons that may exist
 */
class Weapon : public WorldEntity
{
  public:
    // INITIALISATION //
    Weapon (PNode* parent, const Vector& coordinate, const Quaternion& direction);
    Weapon(const TiXmlElement* root);
    virtual ~Weapon ();

    void init();
    void loadParams(const TiXmlElement* root);
    ////////////////////

    void requestAction(WeaponAction action);

    /** @returns true if the Weapon is Active */
    inline bool isActive() const { return this->active; };

    // FUNCTIONS TO SET THE WEAPONS PROPERTIES.
    /** @param projectile a projectile for this weapon */
    void setProjectile(Projectile* projectile) { this->projectile = projectile; };
    /** @returns The projectile if availiable */
    Projectile* getProjectile() { return this->projectile; };

    /** @param state the State to time @param duration the duration of the State */
    inline void setStateDuration(const char* state, float duration) { setStateDuration(charToState(state), duration); };
    /** @param state the State to time @param duration the duration of the State */
    inline void setStateDuration(WeaponState state, float duration) { /*(state < WS_STATE_COUNT)?*/this->times[state] = duration; };
    /** @param state The state to query @returns the Time the queried State takes to complete */
    inline float getStateDuration(WeaponState state) const { return (state < WS_STATE_COUNT)?this->times[state]:0.0; };
    /** @returns true if the time of the currentState is elapsed, false otherwise */
    inline bool stateTimeElapsed() const { return (this->stateDuration > this->times[currentState])?true:false; };
    /** @returns the current State of the Weapon */
    inline WeaponState getCurrentState() const { return this->currentState; };
    /** @param energyMax the maximum energy the Weapon can have @param energyLoadedMax the maximum energy in the weapon buffers */
    inline void setMaximumEnergy(float energyMax, float energyLoadedMax = 0.0) { this->energyMax = energyMax; this->energyLoadedMax = energyLoadedMax; };

    void setActionSound(WeaponAction action, const char* soundFile);
    /** @see void setActionSound(WeaponAction action, const char* soundFile); */
    void setActionSound(const char* action, const char* soundFile) { this->setActionSound(charToAction(action), soundFile); };

    virtual void destroy();


    // FLOW
    virtual void tick(float dt);
    virtual void draw();

    void debug() const;

  protected:
    // utility:
    static WeaponAction  charToAction(const char* action);
    static const char*   actionToChar(WeaponAction action);
    static WeaponState   charToState(const char* state);
    static const char*   stateToChar(WeaponState state);

    //! ACTION: these functions are handled by the Weapon itself, and must be called by requestAction(WeaponAction);
    bool execute();
    virtual void activate();
    virtual void deactivate();
    virtual void fire();
    virtual void reload();
    virtual void charge();

  private:
    bool nextActionValid() const;

  protected:
    SoundSource*         soundSource;
    // it is all about energy
    float                energy;
    float                energyLoaded;
    float                energyMax;
    float                energyLoadedMax;
    float                minCharge;
    float                maxCharge;

    ////////////
    // PHASES //
    ////////////
    WeaponState          currentState;                     //!< The State the weapon is in.
    WeaponAction         requestedAction;                  //!< An action to try to Engage after the currentState ends.
    float                stateDuration;                    //!< how long the state has taken until now.
    float                times[WS_STATE_COUNT];            //!< Times to stay in the different States @see WeaponState.
    Animation3D*         animation[WS_STATE_COUNT];        //!< Animations for all the States (you can say yourself on what part of the gun this animation acts).
    SoundBuffer*         soundBuffers[WA_ACTION_COUNT];    //!< SoundBuffers for all actions @see WeaponAction.


    bool                 hideInactive;                    //!< Hides the Weapon if it is inactive

  private:
    bool                 active;                          //!< states wheter the weapon is enabled or not
    Projectile*          projectile;                      //!< the projectile used for this weapon
  };

#endif /* _WEAPON_H */
