/*!
    \file sound_engine.h
  *  Definition of the SoundEngine singleton Class
*/

#ifndef _SOUND_ENGINE_H
#define _SOUND_ENGINE_H

#include "base_object.h"
#include "alincl.h"


#define SOUND_DOPPLER_FACTOR       0.001          //!< A factor for the audible doppler effect
#define SOUND_DOPPLER_VELOCITY     5000000        //!< A factor for the TravelSpeed of sound

// FORWARD DEFINITION
class PNode;
template<class T> class tList;


//! A class that represents a datastructure to play Sounds.
class SoundBuffer : public BaseObject
{
 public:
  SoundBuffer(const char* fileName);
  ~SoundBuffer();

  /** @returns the ID of the buffer used in this SoundBuffer */
  inline ALuint getID() const { return this->bufferID; }

 private:
  ALuint        bufferID;             //!< The address of the Buffer.

  ALsizei       size;                 //!< The size of the Buffer.
  ALboolean     loop;                 //!< loop information.
};

//! A class that represents a SoundSource
class SoundSource : public BaseObject
{
 public:
  SoundSource(const PNode* sourceNode = NULL, const SoundBuffer* buffer = NULL);
  ~SoundSource();

  // user interaction
  void play();
  void play(const SoundBuffer* buffer);
  void stop();
  void pause();
  void rewind();

  // development functions
  /** @returns The ID of this Source */
  inline ALuint getID() const { return this->sourceID; }
  /** @returns the SoundBuffer of this Source */
  inline const SoundBuffer* getBuffer() const { return this->buffer; }
  /** @returns the SourceNode of this Source */
  inline const PNode* getNode() const { return this->sourceNode;}

  void setRolloffFactor(ALfloat rolloffFactor);

 private:
  ALuint                 sourceID;              //!< The ID of the Source
  const SoundBuffer*     buffer;                //!< The buffer to play in this source.
  const PNode*           sourceNode;            //!< The SourceNode represente the position/velocity... of this source.
};



//! A class that handles audio via the openAudioLibrary
class SoundEngine : public BaseObject {

 public:
  virtual ~SoundEngine();
  /** @returns a Pointer to the only object of this Class */
  inline static SoundEngine* getInstance() { if (!singletonRef) singletonRef = new SoundEngine();  return singletonRef; };

  SoundSource* createSource(const char* fileName, PNode* sourceNode = NULL);

  void setListener(PNode* listener);
  void setDopplerValues(ALfloat dopplerFactor, ALfloat dopplerVelocity);


  void addBuffer(SoundBuffer* buffer);
  void removeBuffer(SoundBuffer* buffer);
  void addSource(SoundSource* source);
  void removeSource(SoundSource* source);

  void update();

  // administrative
  void flushUnusedBuffers();
  void flushAllBuffers();
  void flushAllSources();
  bool initAudio();

  // error handling:
  static void PrintALErrorString(ALenum err);
  //  static void PrintALCErrorString(ALenum err);


 private:
  SoundEngine();
  static SoundEngine*      singletonRef;             //!< Reference to this class


  PNode*                   listener;                 //!< The listener of the Scene
  tList<SoundBuffer>*      bufferList;               //!< A list of buffers
  tList<SoundSource>*      sourceList;               //!< A list for all the sources in the scene.

};

#endif /* _SOUND_ENGINE_H */
