/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
   co-programmer: Benjamin Grauer: injected ResourceManager/GraphicsEngine/GUI
*/

#include "orxonox.h"

#include "gui.h"

#include "world.h"
#include "ini_parser.h"
#include "game_loader.h"

//ENGINES
#include "graphics_engine.h"
#include "sound_engine.h"
#include "resource_manager.h"
#include "object_manager.h"
#include "cd_engine.h"
#include "text_engine.h"
#include "event_handler.h"
#include "garbage_collector.h"

#include "factory.h"
#include "benchmark.h"

#include "class_list.h"
#include "substring.h"

#include <string.h>

int verbose = 4;

using namespace std;

/**
 *  create a new Orxonox

   In this funcitons only global values are set. The game will not be started here.
*/
Orxonox::Orxonox ()
{
  this->setClassID(CL_ORXONOX, "Orxonox");
  this->setName("orxonox-main");

  this->iniParser = NULL;

  this->argc = 0;
  this->argv = NULL;

  this->configFileName = NULL;
}

/**
 *  remove Orxonox from memory
*/
Orxonox::~Orxonox ()
{
  delete this->iniParser;
  delete this->configFileName;

  delete GraphicsEngine::getInstance(); // deleting the Graphics
  delete TextEngine::getInstance();
  delete SoundEngine::getInstance();
  delete ResourceManager::getInstance(); // deletes the Resource Manager
  delete ObjectManager::getInstance();
  delete TextEngine::getInstance();
  delete Factory::getFirst();
  delete GameLoader::getInstance();
  delete SoundEngine::getInstance();
  delete CDEngine::getInstance();
  delete GarbageCollector::getInstance();

  delete EventHandler::getInstance();

  ClassList::debug(3, CL_PARENT_NODE);

  PRINT(3)
      ("===================================================\n" \
      "Thanks for playing orxonox.\n" \
      "visit: http://www.orxonox.ethz.ch for new versions.\n" \
      "===================================================\n");

  Orxonox::singletonRef = NULL;
}

/**
 *  this is a singleton class to prevent duplicates
 */
Orxonox* Orxonox::singletonRef = NULL;

/**
 *  this finds the config file
 * @returns the new config-fileName
 * Since the config file varies from user to user and since one may want to specify different config files
 * for certain occasions or platforms this function finds the right config file for every occasion and stores
 * it's path and name into configfilename
*/
const char* Orxonox::getConfigFile ()
{
  this->configFileName = new char[strlen(DEFAULT_CONFIG_FILE)];
  strcpy (this->configFileName, DEFAULT_CONFIG_FILE);
  this->iniParser = new IniParser(this->configFileName);
}

/**
 * initialize Orxonox with command line
 */
int Orxonox::init (int argc, char** argv)
{
  this->argc = argc;
  this->argv = argv;
  // parse command line
  // config file

  // initialize the Config-file
  this->getConfigFile();


  // initialize everything
  SDL_Init (SDL_INIT_TIMER);
  if( initResources () == -1) return -1;
  if( initVideo() == -1) return -1;
  if( initSound() == -1) return -1;
  if( initInput() == -1) return -1;
  if( initNetworking () == -1) return -1;

  return 0;
}

/**
 * initializes SDL and OpenGL
*/
int Orxonox::initVideo()
{
  PRINTF(3)("> Initializing video\n");

  GraphicsEngine::getInstance();
  GraphicsEngine::getInstance()->setWindowName(PACKAGE_NAME " " PACKAGE_VERSION, PACKAGE_NAME " " PACKAGE_VERSION);

  GraphicsEngine::getInstance()->initFromIniFile(this->iniParser);

  return 0;
}

/**
 * initializes the sound engine
 */
int Orxonox::initSound()
{
  PRINT(3)("> Initializing sound\n");
  // SDL_Init(SDL_INIT_AUDIO);
  SoundEngine::getInstance()->initAudio();
  return 0;
}


/**
 * initializes input functions
 */
int Orxonox::initInput()
{
  PRINT(3)("> Initializing input\n");

  EventHandler::getInstance()->init(this->iniParser);
  EventHandler::getInstance()->subscribe(GraphicsEngine::getInstance(), ES_ALL, EV_VIDEO_RESIZE);


  return 0;
}


/**
 * initializes network system
 */
int Orxonox::initNetworking()
{
  PRINT(3)("> Initializing networking\n");

  printf("  ---Not yet implemented-FIXME--\n");
  return 0;
}


/**
 * initializes and loads resource files
 */
int Orxonox::initResources()
{
  PRINTF(3)("> Initializing resources\n");

  PRINT(3)("initializing ResourceManager\n");

  // create parser
  if( this->iniParser->getSection (CONFIG_SECTION_DATA) == -1)
  {
    PRINTF(1)("Could not find Section %s in %s\n", CONFIG_SECTION_DATA, DEFAULT_CONFIG_FILE);
    return -1;
  }
  char namebuf[256];
  char valuebuf[256];
  memset (namebuf, 0, 256);
  memset (valuebuf, 0, 256);

  while( this->iniParser->nextVar (namebuf, valuebuf) != -1)
  {
    if (!strcmp(namebuf, CONFIG_NAME_DATADIR))
    {
          //  printf("Not yet implemented\n");
      if (!ResourceManager::getInstance()->setDataDir(valuebuf))
      {
        PRINTF(1)("Data Could not be located\n");
        exit(-1);
      }
    }

    memset (namebuf, 0, 256);
    memset (valuebuf, 0, 256);
  }

  if (!ResourceManager::getInstance()->checkDataDir(DEFAULT_DATA_DIR_CHECKFILE))
  {
    PRINTF(1)("The DataDirectory %s could not be verified\n" \
              "  Please Change in File %s Section %s Entry %s to a suitable value\n",
              ResourceManager::getInstance()->getDataDir(),
              DEFAULT_CONFIG_FILE,
              CONFIG_SECTION_DATA,
              CONFIG_NAME_DATADIR);
    exit(-1);
  }
   //! @todo this is a hack and should be loadable
  ResourceManager::getInstance()->addImageDir(ResourceManager::getInstance()->getFullName("maps/"));
  ResourceManager::getInstance()->debug();

  PRINT(3)("initializing TextEngine\n");
  TextEngine::getInstance();

  PRINT(3)("initializing ObjectManager\n");
  ObjectManager::getInstance();
  CDEngine::getInstance();

  return 0;
}

/**
 *  starts the orxonox game or menu
 * here is the central orxonox state manager. There are currently two states
 * - menu
 * - game-play
 * both states manage their states themselfs again.
*/
void Orxonox::start()
{

  this->gameLoader = GameLoader::getInstance();
  this->gameLoader->loadCampaign("worlds/DefaultCampaign.oxc");
  //  this->gameLoader->loadDebugCampaign(DEBUG_CAMPAIGN_0);
  this->gameLoader->init();
  this->gameLoader->start();
}


/**
 * handles sprecial events from localinput
 * @param event: an event not handled by the CommandNode
 */
// void Orxonox::graphicsHandler(SDL_Event* event)
// {
//   // Handle special events such as reshape, quit, focus changes
//   switch (event->type)
//     {
//     case SDL_VIDEORESIZE:
//       GraphicsEngine* tmpGEngine = GraphicsEngine::getInstance();
//       tmpGEngine->resolutionChanged(event->resize);
//       break;
//     }
// }






bool showGui = false;



/**********************************
*** ORXONOX MAIN STARTING POINT ***
**********************************/
/**
 *
 *  main function
 *
 * here the journey begins
*/
int main(int argc, char** argv)
{
  // here the pre-arguments are loaded, these are needed to go either to orxonx itself, Help, or Benchmark.
  int i;
  for(i = 1; i < argc; ++i)
    {
      if(! strcmp( "--help", argv[i]) || !strcmp("-h", argv[i])) return startHelp(argc, argv);
      else if(!strcmp( "--benchmark", argv[i]) || !strcmp("-b", argv[i])) return startBenchmarks();
      else if(!strcmp( "--gui", argv[i]) || !strcmp("-g", argv[i])) showGui = true;
      //      else PRINTF(2)("Orxonox does not understand the arguments %s\n", argv[i]);
    }

  return startOrxonox(argc, argv);
}



int startHelp(int argc, char** argv)
{
  PRINT(0)("orxonox: starts the orxonox game - rules\n");
  PRINT(0)("usage: orxonox [arg [arg...]]\n\n");
  PRINT(0)("valid options:\n");
  {
    Gui* gui = new Gui(argc, argv);
    gui->printHelp();
    delete gui;
  }
  PRINT(0)(" -b|--benchmark:\t\tstarts the orxonox benchmark\n");
  PRINT(0)(" -h|--help:\t\t\tshows this help\n");
}



/**
 * starts orxonox
 * @param argc parameters count given to orxonox
 * @param argv parameters given to orxonox
 */
int startOrxonox(int argc, char** argv)
{
  // checking for existence of the configuration-files, or if the lock file is still used
  if (showGui ||
      !ResourceManager::isFile(DEFAULT_CONFIG_FILE) ||
      ResourceManager::isFile(DEFAULT_LOCK_FILE))
    {
      if (ResourceManager::isFile(DEFAULT_LOCK_FILE))
        ResourceManager::deleteFile(DEFAULT_LOCK_FILE);

      // starting the GUI
      Gui* gui = new Gui(argc, argv);
      gui->startGui();

      if (! gui->startOrxonox)
        return 0;

      delete gui;
    }

  PRINT(0)(">>> Starting Orxonox <<<\n");

  ResourceManager::touchFile(DEFAULT_LOCK_FILE);

  Orxonox *orx = Orxonox::getInstance();

  if((*orx).init(argc, argv) == -1)
    {
      PRINTF(1)("! Orxonox initialization failed\n");
      return -1;
    }

  orx->start();

  delete orx;
  ResourceManager::deleteFile("~/.orxonox/orxonox.lock");

}
