/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_PLAYER

#include "player.h"

#include "track_manager.h"
#include "objModel.h"
#include "resource_manager.h"

#include "weapon_manager.h"
#include "weapon.h"
#include "test_gun.h"
#include "world.h"

#include "list.h"
#include "stdincl.h"

#include "event_handler.h"

#include "projectile.h"
#include "event.h"


using namespace std;

CREATE_FACTORY(Player);

/**
   \brief creates a new Player
   \param isFree if the player is free
*/
Player::Player()
{
  /*
    this is the debug player - actualy we would have to make a new
     class derivated from Player for each player. for now, we just use
     the player.cc for debug also
  */
  this->init();

  //weapons:
  Weapon* wpRight = new TestGun(this, Vector(-2.6, 0.1, 3.0), Quaternion(), 0);
  Weapon* wpLeft = new TestGun(this, Vector(-2.6, 0.1, -3.0), Quaternion(), 1);

  this->weaponMan->addWeapon(wpRight, W_CONFIG0, W_SLOT0);
  this->weaponMan->addWeapon(wpLeft, W_CONFIG1, W_SLOT1);
  this->weaponMan->addWeapon(wpRight, W_CONFIG2);
  this->weaponMan->addWeapon(wpLeft, W_CONFIG2);
}


/**
   \brief destructs the player, deletes alocated memory
*/
Player::~Player ()
{
  /* do not delete the weapons, they are contained in the pnode tree
     and will be deleted there.
     this only frees the memory allocated to save the list.
  */
  delete this->weaponMan;
}


/**
   \brief creates a new Player from Xml Data
   \param root the xml element containing player data

   \todo add more parameters to load
*/
Player::Player(const TiXmlElement* root)
{
  this->init();
  this->loadParams(root);

  //weapons:
  Weapon* wpRight = new TestGun(this, Vector(-2.6, 0.1, 3.0), Quaternion(), 0);
  Weapon* wpLeft = new TestGun(this, Vector(-2.6, 0.1, -3.0), Quaternion(), 1);

  this->weaponMan->addWeapon(wpRight, W_CONFIG0, W_SLOT0);
  this->weaponMan->addWeapon(wpLeft, W_CONFIG1, W_SLOT1);
  this->weaponMan->addWeapon(wpRight, W_CONFIG2);
  this->weaponMan->addWeapon(wpLeft, W_CONFIG2);
}

/**
 * initializes a Player
 */
void Player::init()
{
  this->setClassID(CL_PLAYER, "Player");

  this->model = (Model*)ResourceManager::getInstance()->load("models/reaplow.obj", OBJ, RP_CAMPAIGN);
  travelSpeed = 15.0;
  velocity = new Vector();
  bUp = bDown = bLeft = bRight = bAscend = bDescend = false;
  bFire = false;
  this->bWeaponChange = false;
  acceleration = 10.0;


  this->weaponMan = new WeaponManager();
}


/**
 *
 * @param root the XML-element to load the Player's properties from
 */
void Player::loadParams(const TiXmlElement* root)
{
  static_cast<WorldEntity*>(this)->loadParams(root);



}


/**
   \brief adds a weapon to the weapon list of player
   \param weapon to add
*/
void Player::addWeapon(Weapon* weapon)
{
  this->weaponMan->addWeapon(weapon);
}


/**
   \brief removes a weapon from the player
   \param weapon to remove
*/
void Player::removeWeapon(Weapon* weapon)
{
  this->weaponMan->removeWeapon(weapon);
}


/**
   \brief effect that occurs after the player is spawned
*/
void Player::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}


/**
   \brief the action occuring if the player left the game
*/
void Player::leftWorld ()
{}



/**
   \brief if the player is hit, call this function
   \param weapon hit by this weapon
   \param loc ??
*/
void Player::hit (WorldEntity* weapon, Vector* loc)
{
}


/**
    \brief Collision with another Entity has this effect
    \param other the other colider
    \param ownhitflags ??
    \param otherhitflags ??
*/
void Player::collide (WorldEntity* other, Uint32 ownhitflags, Uint32 otherhitflags)
{
}


/**
   \brief draws the player after transforming him.
*/
void Player::draw ()
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  float matrix[4][4];

  /* translate */
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  /* rotate */
  this->getAbsDir ().matrix (matrix);
  glMultMatrixf((float*)matrix);

  this->model->draw();
  glPopMatrix();

  this->weaponMan->draw();
}


/**
   \brief the function called for each passing timeSnap
   \param time The timespan passed since last update
*/
void Player::tick (float time)
{
  //printf("%p\n", this);
  //this->getRelCoor().debug();

  /* link tick to weapon */
  //this->activeWeapon->tick(time);
  //this->activeWeaponL->tick(time); //FIX FIX DELETE REMOVE
  this->weaponMan->tick(time);
  // player controlled movement
  this->move(time);
  // weapon system manipulation
  this->weapon();
}


/**
   \brief action if player moves
   \param time the timeslice since the last frame
*/
void Player::move (float time)
{
  Vector accel(0.0, 0.0, 0.0);
  /* FIXME: calculating the direction and orthDirection every timeSlice is redundant! save it somewhere */
  /* calculate the direction in which the craft is heading  */
  Vector direction (1.0, 0.0, 0.0);
  //direction = this->absDirection.apply (direction);
  Vector orthDirection (0.0, 0.0, 1.0);
  //orthDirection = orthDirection.cross (direction);

  if( this->bUp && this->getRelCoor().x < 20)
    accel = accel+(direction*acceleration);
  if( this->bDown && this->getRelCoor().x > -5)
    accel = accel -(direction*acceleration);
  if( this->bLeft &&  TrackManager::getInstance()->getWidth() > -this->getRelCoor().z*2)
    accel = accel - (orthDirection*acceleration);
  if( this->bRight &&  TrackManager::getInstance()->getWidth() > this->getRelCoor().z*2)
    accel = accel + (orthDirection*acceleration);
  if( this->bAscend )
  if( this->bDescend) {/* FIXME */} /* \todo up and down player movement */

  Vector move = accel * time;
  this->shiftCoor (move);
}


/**
   \brief weapon manipulation by the player
*/
void Player::weapon()
{
  if( this->bFire)
    {
      this->weaponMan->fire();
    }
  if( this->bWeaponChange)
    {
      this->weaponMan->nextWeaponConf();
      this->bWeaponChange = false;
    }
}


/**
   \brief The connection to the command node
   \param cmd the Command unit from witch to map

   here the commands are mapped to the players movement/weaponary
*/
void Player::command (Command* cmd)
{
  PRINTF(3)("recieved command [%s]\n", cmd->cmd);
  if( !strcmp( cmd->cmd, CONFIG_NAME_PLAYER_UP)) this->bUp = !cmd->bUp;
  if( !strcmp( cmd->cmd, CONFIG_NAME_PLAYER_DOWN)) this->bDown = !cmd->bUp;
  if( !strcmp( cmd->cmd, CONFIG_NAME_PLAYER_LEFT)) this->bLeft = !cmd->bUp;
  if( !strcmp( cmd->cmd, CONFIG_NAME_PLAYER_RIGHT)) this->bRight = !cmd->bUp;
  if( !strcmp( cmd->cmd, CONFIG_NAME_PLAYER_FIRE)) this->bFire = !cmd->bUp;
  if( !strcmp( cmd->cmd, CONFIG_NAME_PLAYER_NEXT_WEAPON)) if(cmd->bUp) this->bWeaponChange = !this->bWeaponChange;
}

/**
 * @todo switch statement ??
 */
void Player::process(const Event &event)
{
  if( event.type == KeyMapper::PEV_UP)
    {
      this->bUp = event.bPressed;
    }
  else if( event.type == KeyMapper::PEV_DOWN)
    {
      this->bDown = event.bPressed;
    }
  else if( event.type == KeyMapper::PEV_RIGHT)
    {
      this->bRight= event.bPressed;
    }
  else if( event.type == KeyMapper::PEV_LEFT)
    {
      this->bLeft = event.bPressed;
    }
  else if( event.type == KeyMapper::PEV_FIRE1)
    {
       this->bFire = event.bPressed;
    }
  else if( event.type == KeyMapper::PEV_NEXT_WEAPON)
    {
      if( !event.bPressed) this->bWeaponChange = !this->bWeaponChange;
    }

}
