/*!
    \file weapon.h
    \brief a weapon that a player can use

    A Player has a list of weapons, that can be choosen to shoot projectiles
    (projectiles.{cc,h}) at ennemies. These weapons can be shooted sequentially
    or (if able) combined. Therefore you can choose the weapon mode = choose
    a weapon.

    A weapon is characterized by:
     o firing-rate: the initial firing rate of a weapon (1/s = Herz)
     o slowdown-factor: this is a factor d: exp(-d*x), d is element of all positive R. it determines how fast the firing-rate will slow down. if no slowdown: d=0, the bigger d is, the faster the weapon will slow down!
     o energy-consumption: this determines the energy that has to be used to produce this projectile = costs per projectile

    Furthermore there are some other attributes, that will help to represent a firing
    weapon in this world:
     o sound file/ressource: this is a pointer to the sound-file/ressource. however it may be represented
     o shooting animation


     a player defines one or more weapon configurations. a player has got one to eight
     weapon slots: places where weapons can be attached to. a weapon configuration
     is a matching between weapons and slots.
     Since its clear how many weapons a player will have, there is no list of weapons:
     its hard coded and limited to 8 slots and 4 configs. More would be a waste of
     memory and time you need to customize and change to a weapon config...
 */

#include "base_object.h"


// FORWARD DECLARATION
class Weapon;

#define    W_MAX_SLOTS       8
#define    W_MAX_CONFIGS     4

//! this is an identifier for the weapon config
#define    W_CONFIG0     0
#define    W_CONFIG1     1
#define    W_CONFIG2     2
#define    W_CONFIG3     3


//! this is an identifier for the slot. there are up to 8 weapon slots -> this means there can't be more than 8 weapons at the same time
#define    W_SLOT0       0
#define    W_SLOT1       1
#define    W_SLOT2       2
#define    W_SLOT3       3
#define    W_SLOT4       4
#define    W_SLOT5       5
#define    W_SLOT6       6
#define    W_SLOT7       7
#define    W_FREE_SLOT   99


//! this is a weapon Configuration: it has up to 8 slots
typedef struct weaponConfig {
  bool           bUsed;                       //!< is set to true, if this configuration is
  Weapon*        slots[8];
};


class WeaponManager : public BaseObject {
  public:
    WeaponManager(int nrOfSlots = 2);
    WeaponManager(const TiXmlElement* root);
    ~WeaponManager();

    void init();
    void loadParams(const TiXmlElement* root);

    void addWeapon(Weapon* weapon, int configID = W_CONFIG0, int slotID = W_FREE_SLOT);
    void removeWeapon(Weapon* weapon, int configID = W_CONFIG0);
    void nextWeaponConf();

    void fire();
    void tick(float sec);
    void draw();

  private:
    int                   nrOfSlots;               //<! number of weapon slots a ship has
    int                   currConfID;              //<! the currently selected config
    weaponConfig          configs[4];              //<! a list of four configurations

    int getNextFreeSlot(int configID);
};
