/*!
    \file weapon.h
    \brief a weapon that a  can use


    A weapon is characterized by:
     o firing-rate: the initial firing rate of a weapon (1/s = Herz)
     o slowdown-factor: this is a factor d: exp(-d*x), d is element of all positive R. it determines how fast the firing-rate will slow down. if no slowdown: d=0, the bigger d is, the faster the weapon will slow down!
     o energy-consumption: this determines the energy that has to be used to produce this projectile = costs per projectile

    Furthermore there are some other attributes, that will help to represent a firing
    weapon in this world:
     o sound file/ressource: this is a pointer to the sound-file/ressource. however it may be represented
     o shooting animation
*/


#ifndef _WEAPON_H
#define _WEAPON_H

#include "base_object.h"
#include "world_entity.h"

// FORWARD DECLARATION
class Projectile;
class Weapon;
class Animation3D;
class TiXmlElement;

//! An enumerator defining actions a Weapon can take
typedef enum {
  WA_NONE          =    0,    //!< No Action taken
  WA_SHOOT         =    1,    //!< emitting Shot
  WA_CHARGE        =    2,    //!< charge action (one click before the shot)
  WA_RELOAD        =    3,    //!< reload right after shoot is finished
  WA_ACTIVATE      =    4,    //!< activate the GUN
  WA_DEACTIVATE    =    5,    //!< deactivate the GUN
  WA_SPECIAL1      =    6,    //!< Special Action taken

  WA_ACTION_COUNT  =    7     //!< This must match the count of enumerations-members.
} WeaponActions;

//! An enumerator defining the States of a Weapon
typedef enum {
  WS_NONE          =    0,    //!< No State at all (if set, there is something wrong, or the weapon is not yet availiable)
  WS_SHOOTING      =    1,    //!< The State of the Shooting
  WS_RELOADING     =    3,    //!< The State of the Reloading
  WS_ACTIVATING    =    4,    //!< The State in which the weapon gets activated
  WS_DEACTIVATING  =    5,    //!< The State in which the weapon gets deactivated
  WS_INACTIVE      =    6,    //!< The State where the weapon is inactive (unable to shoot)
  WS_IDLE          =    7,    //!< The State where the weapon is idle

  WS_STATE_COUNT  =     8      //!< This must match the count of the enumerations-members.
} WeaponState;


//! a weapon can be left or right sided
/**
 * @todo this will be reset with mirror X/Y/Z
 */
#define    W_LEFT        0
#define    W_RIGHT       1

//! An abstract class, that describes weapons
/**
 * This is used as a container for all the different kinds of weapons that may exist
 */
class Weapon : public WorldEntity
{
  friend class World;

 public:
  Weapon (PNode* parent, const Vector& coordinate, const Quaternion& direction);
  Weapon(const TiXmlElement* root);
  virtual ~Weapon ();

  void init();
  void loadParams(const TiXmlElement* root);

  void enable();
  void disable();
  bool isEnabled();

  void setProjectile(Projectile* projectile);
  Projectile* getProjectile();

  virtual void activate();
  virtual void deactivate();
  bool isActive();


  /** @param idle time in ms  */
  inline void setWeaponIdleTime(float idleTime) { this->idleTime = idleTime; };
  /** @returns idle time in ms */
  inline float getWeaponIdleTime() const { return this->idleTime; };
  /** @return true if idletime is elapsed else otherwise */
  inline bool hasWeaponIdleTimeElapsed() const { return (this->localTime>this->idleTime)?true:false; };

  /** @brief fires the weapon */
  virtual void fire() = 0;
  virtual void hit (WorldEntity* weapon, const Vector& loc);
  virtual void destroy();

  virtual void tick(float time);
  virtual void weaponIdle();
  virtual void draw();

 protected:

  float                localTime;                        //<! this is the local time. important for shooting attributes like frequency
  float                idleTime;                         //<! the time a weapon needs before it can shoot again. eg. shooting frequency or actication/deactivateion delay
  float                slowDownFactor;                   //<! if the shooting frequency is a linear function of time...

  ////////////
  // PHASES //
  ////////////
  float                times[WS_STATE_COUNT];            //!< Times to stay in the different States @see WeaponState
  SoundBuffer*         soundBuffers[WA_ACTION_COUNT];    //!< SoundBuffers for all actions @see WeaponAction
  Animation3D*         animation[WS_STATE_COUNT];        //!< Animations for all the States (you can say yourself on what part of the gun this animation acts).


  SoundBuffer*         fireSound;
  SoundSource*         weaponSource;


 private:
   WeaponState          state;                           //!< The state this weapon is in.
   bool                 enabled;                         //<! states if the weapon is enabled or not
   Projectile*          projectile;                      //<! the projectile used for this weapon
  //WeaponSound sound;
};

#endif /* _WEAPON_H */
