
/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific
   main-programmer: Patrick Boenzli
   co-programmer:
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WEAPON

#include "weapon_manager.h"
#include "weapon.h"
#include "stdincl.h"
#include "world_entity.h"
#include "vector.h"
#include "model.h"
#include "projectile.h"
#include "list.h"
#include "world.h"

using namespace std;



/**
 * @brief this initializes the weaponManager for a given nnumber of weapon slots
 * @param number of weapon slots of the model/ship <= 8 (limitied)
 */
              WeaponManager::WeaponManager(int nrOfSlots)
{
  this->setClassID(CL_WEAPON_MANAGER, "WeaponManager");

  for(int i = 0; i < W_MAX_CONFIGS; ++i)
  {
    this->configs[i].bUsed = false;
    for(int j = 0; j < W_MAX_SLOTS; ++j)
      this->configs[i].slots[j] = NULL;
  }
  this->nrOfSlots = nrOfSlots;
  this->currConfID = W_CONFIG0;
}


WeaponManager::~WeaponManager()
{
  /*
  i dont have to delete the weapons itself, because they are
  worldentities and therefore in the entities list of the world.
  world will clean them up for me
  */
  for(int i = 0; i < W_MAX_CONFIGS; ++i)
  {
    this->configs[i].bUsed = false;
    for(int j = 0; j < W_MAX_SLOTS; ++j)
      this->configs[i].slots[j] = NULL;
  }
}


/**
 * @brief adds a weapon to the selected weaponconfiguration into the selected slot
 * @param the weapon to add
 * @param an identifier for the slot: number between 0..7 if not specified: slotID=next free slot
 * @param an identifier for the weapon configuration, number between 0..3

 if you add explicitly a weapon at config:n, slot:m, the weapon placed at this location will be
              replaced by the weapon specified. if you use the W_FREE_SLOT, the manager will look for a free
              slot in this weaponconfiguration. if there is non, the weapon won't be added and there will be
              a error message.
 */
              void WeaponManager::addWeapon(Weapon* weapon, int configID, int slotID)
{
  if( slotID == W_FREE_SLOT)
  {
    int freeSlot = this->getNextFreeSlot( configID);
    if( freeSlot < 0 || freeSlot >= this->nrOfSlots)
    {
      PRINTF(0)("There is no free slot in this WeaponConfig to dock this weapon at! Aborting\n");
      return;
    }
    PRINTF(3)("Added new Weapon to Config:%i/Slot:%i\n", configID, freeSlot);
    this->configs[configID].bUsed = true;
    this->configs[configID].slots[freeSlot] = weapon;
    return;
  }
  this->configs[configID].bUsed = true;
  this->configs[configID].slots[slotID] = weapon;
  PRINTF(3)("Added a new Weapon to the WeaponManager: config %i/ slot %i\n", configID, slotID);
}


void WeaponManager::removeWeapon(Weapon* weapon, int configID)
{
  /* empty */
}


/**
   * @brief changes to the next weapon configuration

              if there are multiple weapon configurations defined by the manager, use this to switch between them
              this function will deactivate the weapons first, change the config and reactivate them later
 */
              void WeaponManager::nextWeaponConf()
{
  PRINTF(4)("Changing weapon configuration: from %i to next\n", this->currConfID);

  int i, lastConfID;
  lastConfID = this->currConfID;
  for(i = this->currConfID + 1; i < W_MAX_CONFIGS && !this->configs[i].bUsed; ++i);
  if( i == W_MAX_CONFIGS) this->currConfID = W_CONFIG0;
  else this->currConfID = i;


  Weapon *w1, *w2;
  for(int j = 0; j < W_MAX_SLOTS; ++j)
  {
    w1 = this->configs[lastConfID].slots[j];
    w2 = this->configs[this->currConfID].slots[j];

    if( w1 == w2)
    {
      printf("no need for change\n");
    }
    else
    {
      if( w1 != NULL )
      {
        w1->deactivate();
        printf("deactivating %i,%i\n", j,lastConfID);
      }
      if( w2 != NULL)
      {
        w2->activate();
        printf("activating %i,%i\n", j, this->currConfID);
      }
    }
  }
}



/**
   * @brief triggers fire of all weapons in the current weaponconfig
 */
              void WeaponManager::fire()
{
  Weapon* firingWeapon;
  for(int i = 0; i < W_MAX_SLOTS; ++i)
  {
    firingWeapon = this->configs[this->currConfID].slots[i];
    if( firingWeapon != NULL) firingWeapon->fire();
  }
}


/**
   * @brief triggers tick of all weapons in the current weaponconfig
              * @param second passed since last tick
 */
              void WeaponManager::tick(float sec)
{
  Weapon* w;
  for(int i = 0; i < W_MAX_SLOTS; ++i)
  {
    w = this->configs[this->currConfID].slots[i];
    if( w != NULL) w->tick(sec);
  }
}


/**
   * @brief triggers draw of all weapons in the current weaponconfig
 */
void WeaponManager::draw()
{
  Weapon* w;
  for(int i = 0; i < W_MAX_SLOTS; ++i)
  {
    w = this->configs[this->currConfID].slots[i];
    if( w != NULL) w->draw();
  }
}


/**
   * @brief private gets the next free slot in a certain weaponconfig
              * @param the selected weaponconfig
 */
              int WeaponManager::getNextFreeSlot(int configID)
{
  for( int i = 0; i < W_MAX_SLOTS; ++i)
  {
    if( this->configs[configID].slots[i] == NULL)
      return i;
  }
  return -1;
}

