/*!
    \file weapon.h
    \brief a weapon that a player can use


    A weapon is characterized by:
     o firing-rate: the initial firing rate of a weapon (1/s = Herz)
     o slowdown-factor: this is a factor d: exp(-d*x), d is element of all positive R. it determines how fast the firing-rate will slow down. if no slowdown: d=0, the bigger d is, the faster the weapon will slow down!
     o energy-consumption: this determines the energy that has to be used to produce this projectile = costs per projectile

    Furthermore there are some other attributes, that will help to represent a firing
    weapon in this world:
     o sound file/ressource: this is a pointer to the sound-file/ressource. however it may be represented
     o shooting animation
*/


#ifndef _WEAPON_H
#define _WEAPON_H

#include "base_object.h"
#include "world_entity.h"

#define W_MAX_SLOTS 8
#define W_MAX_CONFIGS 4


// FORWARD DECLARATION
class Projectile;
class Weapon;
class Animation3D;
class TiXmlElement;


// typedef enum {
//   W_SHOOT,
//   W_EMPTY,
//   W_RELOAD,
//   W_SPECIAL1,
//   W_SPECIAL2,
//   W_SPECIAL3
// } WeaponSoundType;

typedef enum {
  W_NONE,
  W_SHOOT,
  W_RELOAD,
  W_ACTIVATING,
  W_DEACTIVATE,
  W_IDLE,
} WeaponState;


//! a weapon can be left or right sided
#define    W_LEFT        0
#define    W_RIGHT       1





class Weapon : public WorldEntity
{
  friend class World;

 public:
  Weapon (PNode* parent, const Vector& coordinate, const Quaternion& direction);
  Weapon(const TiXmlElement* root);
  virtual ~Weapon ();

  void init();
  void loadParams(const TiXmlElement* root);

  void enable();
  void disable();
  bool isEnabled();

  void setProjectile(Projectile* projectile);
  Projectile* getProjectile();

  virtual void activate();
  virtual void deactivate();
  bool isActive();


  /**
     \brief sets a weapons idle time
     \param idle time in ms

     a weapon idle time is the time spend after a shoot until the weapon can
     shoot again
  */
  inline void setWeaponIdleTime(float time) { this->idleTime = time; }
  /**
     \brief gets the weapon idle time
     \returns idle time in ms
   */
  inline float getWeaponIdleTime() const { return this->idleTime;}
  /**
     \brief checks if the idle time is elapsed
     \return true if time is elapsed

     a weapon idle time is the time spend after a shoot until the weapon can
   shoot again
  */
  inline bool hasWeaponIdleTimeElapsed() const { return (this->localTime>this->idleTime)?true:false; }

  /**
     \brief fires the weapon

     this is called from the player.cc, when fire-button is been pushed
  */
  virtual void fire() = 0;
  virtual void hit (WorldEntity* weapon, Vector* loc);
  virtual void destroy();

  virtual void tick(float time);
  virtual void weaponIdle();
  virtual void draw();

 protected:
  tList<WorldEntity>* worldEntities;
  float localTime;                 //<! this is the local time. important for shooting attributes like frequency
  float idleTime;                  //<! the time a weapon needs before it can shoot again. eg. shooting frequency or actication/deactivateion delay
  float slowDownFactor;            //<! if the shooting frequency is a linear function of time...

  PNode* objectComponent1;         //<! the gun is made of multiple parts, these PNodes represent their location and orientation
  PNode* objectComponent2;
  PNode* objectComponent3;

  Animation3D* animation1;
  Animation3D* animation2;
  Animation3D* animation3;

  Vector projectileOffset;
  int leftRight;   // this will become an enum

  SoundBuffer* fireSound;
  SoundSource* weaponSource;


 private:
  bool enabled;                    //<! states if the weapon is enabled or not
  Projectile* projectile;          //<! the projectile used for this weapon
  //WeaponSound sound;
};

#endif /* _WEAPON_H */
