/*!
    \file graphics_engine.h

    \brief defines a Interface between orxonox and graphical input

    handles graphical SDL-initialisation, textures, resolutions, and so on
*/

#ifndef _GRAPHICS_ENGINE_H
#define _GRAPHICS_ENGINE_H

#include "base_object.h"

#include "sdlincl.h"
#include "glincl.h"

// Forward Declaration
class Text;
class IniParser;

//! class to handle graphics
/**
   handles graphical SDL-initialisation, textures, resolutions, and so on
 */
class GraphicsEngine : public BaseObject
{
  public:
    virtual ~GraphicsEngine();
    /** \returns a Pointer to the only object of this Class */
    inline static GraphicsEngine* getInstance() { if (!singletonRef) singletonRef = new GraphicsEngine();  return singletonRef; };

    int initVideo();
    int loadFromIniFile(IniParser* iniParser, const char* section);

    void setWindowName(const char* windowName, const char* icon);
    int setGLattribs();
    int setResolution(int width, int height, int bpp);
    void setFullscreen(bool fullscreen = false);
    static void setBackgroundColor(GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha = 1.0);

    /** \returns the x resolution */
    inline int getResolutionX() const { return this->resolutionX; };
    /** \returns the y resolution */
    inline int getResolutionY() const { return this->resolutionY; };
    /** \returns the Bits Per Pixel */
    inline int getbbp() const { return this->bitsPerPixel; };

    int resolutionChanged(SDL_ResizeEvent* resizeInfo);

    static void enter2DMode();
    static void leave2DMode();

    static void storeMatrices();
    static GLdouble modMat[16];
    static GLdouble projMat[16];
    static GLint viewPort[4];

    void tick(float dt);
    void displayFPS(bool display);

    void listModes();

    /** \brief swaps the GL_BUFFERS */
    static void swapBuffers() { SDL_GL_SwapBuffers(); };

  public:
    static bool texturesEnabled;

  private:
    GraphicsEngine();

  private:
    static GraphicsEngine* singletonRef;

    SDL_Surface*   screen;          //!< the screen we draw to
    int            resolutionX;     //!< the X-resoultion of the screen
    int            resolutionY;     //!< the Y-resolution of the screen
    int            bitsPerPixel;    //!< the bits per pixels of the screen
    Uint32         fullscreenFlag;  //!< if we are in fullscreen mode
    Uint32         videoFlags;      //!< flags for video
    SDL_Rect**     videoModes;      //!< list of resolutions

    bool           bDisplayFPS;     //!< is true if the fps should be displayed
    float          currentFPS;      //!< the current frame rate: frames per seconds
    float          maxFPS;          //!< maximal frame rate we ever got since start of the game
    float          minFPS;          //!< minimal frame rate we ever got since start


#ifndef NO_TEXT
  Text*          geTextCFPS;      //!< Text for the current FPS
  Text*          geTextMaxFPS;    //!< Text for the max FPS
  Text*          geTextMinFPS;    //!< Text for the min FPS
#endif /* NO_TEXT */
};

#endif /* _GRAPHICS_ENGINE_H */
