

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: ...
*/


#include "base_object.h"

#include "load_param.h"
#include "compiler.h"
#include "class_list.h"

using namespace std;


/**
   \brief sets the name from a LoadXML-Element
   \param root the element to load from
*/
BaseObject::BaseObject(const TiXmlElement* root)
{
  this->className = NULL;
  this->classID = CL_BASE_OBJECT;
  this->finalized = false;

  this->objectName = NULL;

  if (root)
    this->loadParams(root);

  ClassList::addToClassList(this, this->classID, "BaseObject");
}

/**
   \brief standard deconstructor
*/
BaseObject::~BaseObject ()
{
  ClassList::removeFromClassList(this);

  //  delete []this->className;
  if (this->objectName)
    delete []this->objectName;}

/**
   \brief loads parameters
   \param root the element to load from
*/
void BaseObject::loadParams(const TiXmlElement* root)
{
  // name setup
  LoadParam<BaseObject>(root, "name", this, &BaseObject::setName)
  .describe("the name of the Object at hand");
}

/**
   \brief sets the class identifiers
   \param id a number for the class from class_id.h enumeration
   \param className the class name
*/
void BaseObject::setClassID(long classID, const char* className)
{
  this->classID |= classID;
  this->className = className;

  ClassList::addToClassList(this, classID, className);
}

/**
  \brief set the name of the Object
 */
void BaseObject::setName (const char* objectName)
{
  if (this->objectName)
    delete []this->objectName;
  if (objectName)
  {
    this->objectName = new char[strlen(objectName)+1];
    strcpy(this->objectName, objectName);
  }
  else
    this->objectName = NULL;
}


/**
   \brief checks if the class is a classID
   \param classID the Identifier to check for
   \returns true if it is, false otherwise
*/
bool BaseObject::isA (long classID) const
{
  // if classID is a derivable object
  if (likely(classID & CL_MASK_SUPER_CLASS || classID & CL_MASK_SUBSUPER_CLASS))
  {
    if( this->classID & classID)
      return true;
  }
  // if classID is a LOWLEVEL-class
  else
  {
    if ((this->classID & CL_MASK_LOWLEVEL_CLASS) == classID)
      return true;
  }
  return false;
}

/**
 * @brief displays everything this class is
 */
void BaseObject::whatIs() const
{
  PRINT(0)("object %s of class %s: ", this->getName(), this->getClassName());
  if ((this->classID & CL_MASK_SINGLETON) == CL_MASK_SINGLETON)
    PRINT(0)("is a Singleton-Class ");
  if (this->classID & CL_MASK_SUPER_CLASS)
  {
    PRINT(0)(" ->is a derived from the following superclasses:");
    if (this->isA(CL_BASE_OBJECT))
      PRINT(0)(" =BaseObject=");
    if (this->isA(CL_PARENT_NODE))
      PRINT(0)(" =PNode=");
    if (this->isA(CL_WORLD_ENTITY))
      PRINT(0)(" =WorldEntity=");
    if (this->isA(CL_PHYSICS_INTERFACE))
      PRINT(0)(" =PhysicsInterface=");
    if (this->isA(CL_EVENT_LISTENER))
      PRINT(0)(" =EventListener=");
    if (this->isA(CL_STORY_ENTITY))
      PRINT(0)(" =StoryEntity=");
    PRINT(0)("\n");
  }
  // subsuper-classes
  if (this->classID & CL_MASK_SUBSUPER_CLASS)
  {
    PRINT(0)(" ->further derivations: ");
    if (this->isA(CL_PLAYER))
      PRINT(0)(" -Player-");
    if (this->isA(CL_NPC))
      PRINT(0)(" -NPC-");
    if (this->isA(CL_POWER_UP))
      PRINT(0)(" -PowerUp-");
    if (this->isA(CL_FIELD))
      PRINT(0)(" -Field-");
    if (this->isA(CL_PROJECTILE))
      PRINT(0)(" -Projectile-");
    if (this->isA(CL_WEAPON))
      PRINT(0)(" -Weapon-");
    PRINT(0)("\n");
  }
}
