/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: ...
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_PHYSICS

#include "physics_engine.h"

#include "debug.h"

#include "list.h"
#include "tinyxml.h"
#include "factory.h"

using namespace std;


/**
   \brief standard constructor
*/
PhysicsEngine::PhysicsEngine(void)
{
  this->setClassID(CL_PHYSICS_ENGINE, "PhysicsEngine");
  this->setName("PhysicsEngine");
  this->connections = new tList<PhysicsConnection>;
  this->interfaces = new tList<PhysicsInterface>;
  this->fields = new tList<Field>;
}

/**
   \brief the singleton reference to this class
*/
PhysicsEngine* PhysicsEngine::singletonRef = NULL;

/**
   \brief standard deconstructor

*/
PhysicsEngine::~PhysicsEngine(void)
{
  PhysicsEngine::singletonRef = NULL;
}

/**
  \param root the XML-element to load settings from
 */
void PhysicsEngine::loadParams(const TiXmlElement* root)
{
  const TiXmlElement* element = NULL;

  PRINTF(0)("Loading Physical Fields\n");
  element = root->FirstChildElement("Fields");
  printf("PPPPPOOOOINNNTERRRR: %p\n", element);
  element = element->FirstChildElement();
  printf("PPPPPOOOOINNNTERRRR: %p\n", element);
  while (element != NULL)
  {
    Factory::getFirst()->fabricate(element)->getName();

    element = element->NextSiblingElement();
  }
  element = NULL;

  PRINTF(0)("Loading Physical Connections\n");
  element = root->FirstChildElement("Connections");
  element = element->FirstChildElement();
  while (element != NULL)
  {
    Factory::getFirst()->fabricate(element);

    element = element->NextSiblingElement();
  }
}

/**
   \brief adds a PhysicsInterface to the list of handeled physicsInterfaces
   \param physicsInterface the interface to add

   this is normally done in the constructor of any PhysicsInterface
*/
void PhysicsEngine::addPhysicsInterface(PhysicsInterface* physicsInterface)
{
  this->interfaces->add(physicsInterface);
}

/**
   \brief removes a PhysicsInterface from the list of handeled physicsInterfaces
   \param physicsInterface the interface to remove

   this is normally done in the destructor of any PhysicsInterface
*/
void PhysicsEngine::removePhysicsInterface(PhysicsInterface* physicsInterface)
{
  this->interfaces->remove(physicsInterface);
}

/**
  \param physicsInterfaceName the Name of the PhysicsInterface to search for
  \returns the PhysicsInterface if found, or NULL if not
 */
PhysicsInterface* PhysicsEngine::getPhysicsInterfaceByName(const char* physicsInterfaceName) const
{
  tIterator<PhysicsInterface>* tmpIt = interfaces->getIterator();
  PhysicsInterface* tmpInt = tmpIt->nextElement();
  while(tmpInt)
  {
    if (!strcmp(physicsInterfaceName, tmpInt->getName()))
    {
      delete tmpIt;
      return tmpInt;
    }
    tmpInt = tmpIt->nextElement();
  }
  delete tmpIt;
  return NULL;
}

/**
   \brief adds a Field to the list of handeled fields
   \param field the field to add

   this is normally done in the constructor of any Field
*/
void PhysicsEngine::addField(Field* field)
{
  this->fields->add(field);
}

/**
   \brief removes a Field from the list of handeled fields
   \param field the field to remove

   this is normally done in the destructor of any Field
*/
void PhysicsEngine::removeField(Field* field)
{
  this->fields->remove(field);
}

/**
  \param FieldName the Name of the PhysicsInterface to search for
  \returns the Field if found, or NULL if not
 */
Field* PhysicsEngine::getFieldByName(const char* FieldName) const
{
  tIterator<Field>* tmpIt = fields->getIterator();
  Field* tmpField = tmpIt->nextElement();
  while(tmpField)
  {
    if (!strcmp(FieldName, tmpField->getName()))
    {
      delete tmpIt;
      return tmpField;
    }
    tmpField = tmpIt->nextElement();
  }
  delete tmpIt;
  return NULL;
}



/**
   \brief adds A Physical Connection to the List of Connections
   \param connection the Connection to add

   Usually this is done through the constructor of PhysicshConnections
*/
void PhysicsEngine::addConnection(PhysicsConnection* connection)
{
  this->connections->add(connection);
}

/**
   \brief removes A Physical Connection from the List of Connections
   \param connection the Connection to remove

   Usually this is done through the destructor of PhysicsConnections
*/
void PhysicsEngine::removeConnection(PhysicsConnection* connection)
{
  this->connections->remove(connection);
}

/**
  \param physicsConnectionName the Name of the PhysicsInterface to search for
  \returns the PhysicsConnection if found, or NULL if not
 */
PhysicsConnection* PhysicsEngine::getPhysicsConnectionByName(const char* physicsConnectionName) const
{
  tIterator<PhysicsConnection>* tmpIt = connections->getIterator();
  PhysicsConnection* tmpConn = tmpIt->nextElement();
  while(tmpConn)
  {
    if (!strcmp(physicsConnectionName, tmpConn->getName()))
    {
      delete tmpIt;
      return tmpConn;
    }
    tmpConn = tmpIt->nextElement();
  }
  delete tmpIt;
  return NULL;
}



/**
   \brief Steps through all the Connections and Ticks them
   \param dt The time Passed in Seconds

   This function brings a flow into the whole animation
*/
void PhysicsEngine::tick(float dt)
{
  /* go through all the PhysicsInterface(s) and tick them,
  meaning let the fields work */
  tIterator<PhysicsConnection>* itPC = this->connections->getIterator();
  PhysicsConnection* enumPC = itPC->nextElement();
  while (enumPC)
    {
      enumPC->apply();

      enumPC = itPC->nextElement();
    }
  delete itPC;

  /* actually tick all the PhysicsInterfaces. Move the objects around */
  tIterator<PhysicsInterface>* itPI = this->interfaces->getIterator();
  PhysicsInterface* enumPI = itPI->nextElement();
  while (enumPI)
    {
      enumPI->tickPhys(dt);

      enumPI = itPI->nextElement();
    }
  delete itPI;
}



/**
   \brief print out interesting debug information of this class
*/
void PhysicsEngine::debug(void) const
{
  PRINT(0)("====================================\n");
  PRINT(0)("= Physics-Engine debug information =\n");
  PRINT(0)("====================================\n");
  PRINT(0)(" reference: %p\n", this);
  PRINT(0)(" number of Interfaces: %d\n", this->interfaces->getSize());
  PRINT(0)(" number of Fields: %d\n", this->fields->getSize());
  PRINT(0)(" number of Connections: %d\n", this->connections->getSize());

  PRINT(0)("==============================PHYS==\n");
}
