

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#include <iostream>

#include "world_entity.h"
#include "list.h"
#include "vector.h"
#include "obb_tree.h"

using namespace std;

/**
   \brief Loads the WordEntity-specific Part of any derived Class
*/
WorldEntity::WorldEntity(const TiXmlElement* root)
{
  this->setClassID(CL_WORLD_ENTITY, "WorldEntity");

  this->model = NULL;
  this->obbTree = NULL;

  if (root)
    this->loadParams(root);

  this->bDraw = true;
}

/**
   \brief standard destructor
*/
WorldEntity::~WorldEntity ()
{
  // if( collisioncluster != NULL) delete collisioncluster;
  if (this->model)
    ResourceManager::getInstance()->unload(this->model);
}

void WorldEntity::loadParams(const TiXmlElement* root)
{
  static_cast<PNode*>(this)->loadParams(root);
  // Model Loading
  LoadParam<WorldEntity>(root, "model", this, &WorldEntity::loadModel)
    .describe("the fileName of the model, that should be loaded onto this world-entity. (must be relative to the data-dir)") ;
}

/**
   \brief loads a Model onto a WorldEntity
   \param fileName the name of the model to load
*/
void WorldEntity::loadModel(const char* fileName)
{
  if (this->model)
    ResourceManager::getInstance()->unload(this->model, RP_LEVEL);
  this->model = (Model*)ResourceManager::getInstance()->load(fileName, OBJ, RP_CAMPAIGN);
}

/**
   \brief sets the character attributes of a worldentity
   \param character attributes

   these attributes don't have to be set, only use them, if you need them
*/
void WorldEntity::setCharacterAttributes(CharacterAttributes* charAttr)
{}


/**
   \brief gets the Character attributes of this worldentity
   \returns character attributes
*/
CharacterAttributes* WorldEntity::getCharacterAttributes()
{}


/**
   \brief set the WorldEntity's collision hull
   \param newhull: a pointer to a completely assembled CollisionCluster

   Any previously assigned collision hull will be deleted on reassignment
*/
/*
void WorldEntity::setCollision (CollisionCluster* newhull)
{
  if( newhull == NULL) return;
  if( collisioncluster != NULL) delete collisioncluster;
  collisioncluster = newhull;
}
*/


/**
    \brief process draw function
*/
void WorldEntity::processDraw ()
{

}

/**
   \brief sets the drawable state of this entity.
   \param bDraw TRUE if draweable, FALSE otherwise
*/
void WorldEntity::setDrawable (bool bDraw)
{
  this->bDraw = bDraw;
}


/**
   \brief this function is called, when two entities collide
   \param entity: the world entity with whom it collides

   Implement behaviour like damage application or other miscellaneous collision stuff in this function
*/
void WorldEntity::collideWith(WorldEntity* entity)
{
  this->obbTree->collideWith(entity->obbTree);

}


/**
   \brief this function is called, when the ship is hit by a waepon
   \param weapon: the laser/rocket/shoot that hits.
   \param loc: place where it is hit

   calculate the damage depending
*/
void WorldEntity::hit(WorldEntity* weapon, Vector* loc) {}


/**
   \brief this is called immediately after the Entity has been constructed and initialized

   Put any initialisation code that requires knowledge of location (placement if the Entity is free) and owner of the entity here.
   DO NOT place such code in the constructor, those variables are set AFTER the entity is constucted.
*/
void WorldEntity::postSpawn ()
{
}


/**
   \brief this method is called by the world if the WorldEntity leaves valid gamespace

   For free entities this means it left the Track boundaries. With bound entities it means its Location adresses a
   place that is not in the world anymore. In both cases you might have to take extreme measures (a.k.a. call destroy).
*/
void WorldEntity::leftWorld ()
{
}


/**
   \brief this method is called every frame
   \param time: the time in seconds that has passed since the last tick

   Handle all stuff that should update with time inside this method (movement, animation, etc.)
*/
void WorldEntity::tick(float time)
{
}


/**
   \brief the entity is drawn onto the screen with this function

   This is a central function of an entity: call it to let the entity painted to the screen. Just override this function with whatever you want to be drawn.
*/
void WorldEntity::draw()
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  float matrix[4][4];

  /* translate */
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  /* rotate */
  this->getAbsDir ().matrix (matrix);
  glMultMatrixf((float*)matrix);

  if (this->model)
    this->model->draw();
  glPopMatrix();
}


void WorldEntity::drawBVTree(int depth, int drawMode)
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  float matrix[4][4];

  /* translate */
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  /* rotate */
  this->getAbsDir ().matrix (matrix);
  glMultMatrixf((float*)matrix);

  if (this->obbTree)
    this->obbTree->drawBV(depth, drawMode);
  glPopMatrix();
}

/**
   \brief this handles incoming command messages
   \param cmd: a pointer to the incoming Command structure

   Put all code that handles Command messages here, this will mainly be called by the assigned CommandNode but can also be used
   to send commands from one WorldEntity to another.
*/
void WorldEntity::command (Command* cmd)
{
}
