/*!
    \file particle_engine.h
    \brief Definition of the ParticleEngine
*/

#ifndef _PARTICLE_ENGINE_H
#define _PARTICLE_ENGINE_H

#include "base_object.h"
#include "particle_system.h"
#include "particle_emitter.h"

// FORWARD DEFINITION
template<class T> class tList;

//! A ParticleConnection enables us to emitt from any emitter into any other particleSystem
typedef struct ParticleConnection
{
  ParticleEmitter*    emitter;     //!< The emitter to emit system from.
  ParticleSystem*     system;      //!< The Particles emitted from emitter.
};

//! The physicsEngine handles and stores Systems and Emitters.
/**
   It is responsible for driving on the Particles (tick)
   It draw particles (draw)
   and it emitts particles into the system
*/
class ParticleEngine : public BaseObject {

 public:
  virtual ~ParticleEngine(void);
  /** \returns a Pointer to the only object of this Class */
  inline static ParticleEngine* getInstance(void) { if (!singletonRef) singletonRef = new ParticleEngine();  return singletonRef; };

  void tick(float dt);
  void draw(void) const;

  void addSystem(ParticleSystem* system);
  void addEmitter(ParticleEmitter* emitter);
  void addConnection(ParticleEmitter* emitter, ParticleSystem* system);

  bool removeSystem(ParticleSystem* system);
  bool removeEmitter(ParticleEmitter* emitter);
  bool breakConnection(ParticleEmitter* emitter, ParticleSystem* system);
  bool breakConnection(ParticleConnection* connection);

  ParticleSystem* getSystemByName(const char* systemName) const;
  ParticleSystem* getSystemByNumber(unsigned int number) const;
  ParticleEmitter* getEmitterByName(const char* emitterName) const;
  ParticleEmitter* getEmitterByNumber(unsigned int number) const;

  void debug();

 private:
  ParticleEngine(void);
  static ParticleEngine* singletonRef;        //!< The reference to the engine.

  tList<ParticleSystem>* systemList;          //!< A list of Systems handled by the ParticleEngine.
  tList<ParticleEmitter>* emitterList;        //!< A list of Emitters handled by the ParticleEngine.

  tList<ParticleConnection>* connectionList;  //!< A list of Connections between Systems and Emitters.
};

#endif /* _PARTICLE_ENGINE_H */
