/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "material.h"

#include "texture.h"
#include "debug.h"
#include "resource_manager.h"
#include <stdlib.h>
#include <string.h>

//! \todo check if we are in RESOURCE MANAGER-mode
#include "resource_manager.h"

using namespace std;

/**
   \brief creates a Material.
   \param mtlName Name of the Material to be added to the Material List
*/
Material::Material (const char* mtlName)
{
   PRINTF(4)("initializing new Material.\n");
  this->name = NULL;
  this->setIllum(3);
  this->setDiffuse(0,0,0);
  this->setAmbient(0,0,0);
  this->setSpecular(.5,.5,.5);
  this->setShininess(2.0);
  this->setTransparency(1.0);

  this->diffuseTexture = NULL;
  this->ambientTexture = NULL;
  this->specularTexture = NULL;

  this->setName(mtlName);
}

/** 
    \brief deletes a Material
*/
Material::~Material()
{
  PRINTF(4)("delete Material %s.\n", this->name);
  if (this->name)
    delete []this->name;
  if (this->diffuseTexture)
    ResourceManager::getInstance()->unload(this->diffuseTexture);
}

/**
   \brief sets the material with which the following Faces will be painted
*/
bool Material::select (void)
{
  // setting diffuse color
  //  glColor3f (diffuse[0], diffuse[1], diffuse[2]);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, this->diffuse);

  // setting ambient color
  glMaterialfv(GL_FRONT, GL_AMBIENT, this->ambient);

  // setting up Sprecular
  glMaterialfv(GL_FRONT, GL_SPECULAR, this->specular);

  // setting up Shininess
  glMaterialf(GL_FRONT, GL_SHININESS, this->shininess);
  
  // setting the transparency
  if (this->transparency < 1.0)
    {
      glEnable(GL_BLEND);
      glColor4f(1.0f, 1.0f, 1.0f, this->transparency);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE);
    }
  else
    {
      glDisable(GL_BLEND);
      glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
    }


  // setting illumination Model 
  if (this->illumModel == 1) //! \todo make this work, if no vertex-normals are read.
    glShadeModel(GL_FLAT);
  else if (this->illumModel >= 2)
    glShadeModel(GL_SMOOTH);

  if (this->diffuseTexture)
    {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, this->diffuseTexture->getTexture());

      /* This allows alpha blending of 2D textures with the scene */
      if (this->diffuseTexture->hasAlpha())
	{
	  glEnable(GL_BLEND);
	  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}
    }
  else
    {
      glDisable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, 0);
    }
}

/**
   \brief Set the Name of the Material. (Important for searching)
   \param mtlName the Name of the Material to be set.
*/ 
void Material::setName (const char* mtlName)
{
  if (this->name)
    delete this->name;
  if (mtlName)
    {
      this->name = new char [strlen(mtlName)+1];
      strcpy(this->name, mtlName);
    }
  else
    {
      this->name = new char[2];
      strcpy(this->name, "");
    }
}

/**
   \returns The Name of The Material
*/
char* Material::getName (void)
{
  return this->name;
}

/**
   \brief Sets the Material Illumination Model. 
   \brief illu illumination Model in int form
*/
void Material::setIllum (int illum)
{
  PRINTF(4)("setting illumModel of Material %s to %i\n", this->name, illum);
  this->illumModel = illum;
}
/**
   \brief Sets the Material Illumination Model. 
   \brief illu illumination Model in char* form
*/void Material::setIllum (char* illum)
{
  this->setIllum (atoi(illum));
}

/**
   \brief Sets the Material Diffuse Color.
   \param r Red Color Channel.
   \param g Green Color Channel.
   \param b Blue Color Channel.
*/
void Material::setDiffuse (float r, float g, float b)
{
  PRINTF(4)("setting Diffuse Color of Material %s to r=%f g=%f b=%f.\n", this->name, r, g, b);
  this->diffuse[0] = r;
  this->diffuse[1] = g;
  this->diffuse[2] = b;  
  this->diffuse[3] = 1.0;

}
/**
   \brief Sets the Material Diffuse Color.
   \param rgb The red, green, blue channel in char format (with spaces between them)
*/
void Material::setDiffuse (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setDiffuse (r, g, b);
}

/**
   \brief Sets the Material Ambient Color. 
   \param r Red Color Channel.
   \param g Green Color Channel.
   \param b Blue Color Channel.
*/
void Material::setAmbient (float r, float g, float b)
{
  PRINTF(4)("setting Ambient Color of Material %s to r=%f g=%f b=%f.\n", this->name, r, g, b);
  this->ambient[0] = r;
  this->ambient[1] = g;
  this->ambient[2] = b;
  this->ambient[3] = 1.0;
}
/**
   \brief Sets the Material Ambient Color.
   \param rgb The red, green, blue channel in char format (with spaces between them)
*/
void Material::setAmbient (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setAmbient (r, g, b);
}

/**
   \brief Sets the Material Specular Color. 
   \param r Red Color Channel.
   \param g Green Color Channel.
   \param b Blue Color Channel.
*/
void Material::setSpecular (float r, float g, float b)
{
  PRINTF(4)("setting Specular Color of Material %s to r=%f g=%f b=%f.\n", this->name, r, g, b);
  this->specular[0] = r;
  this->specular[1] = g;
  this->specular[2] = b;
  this->specular[3] = 1.0;
 }
/**
   \brief Sets the Material Specular Color.
   \param rgb The red, green, blue channel in char format (with spaces between them)
*/
void Material::setSpecular (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setSpecular (r, g, b);
}

/**
   \brief Sets the Material Shininess.
   \param shini stes the Shininess from float.
*/
void Material::setShininess (float shini)
{
  this->shininess = shini;
}
/**
   \brief Sets the Material Shininess.
   \param shini stes the Shininess from char*.
*/
void Material::setShininess (char* shini)
{
  this->setShininess (atof(shini));
}

/**
   \brief Sets the Material Transparency.
   \param trans stes the Transparency from int.
*/
void Material::setTransparency (float trans)
{
  PRINTF(4)("setting Transparency of Material %s to %f.\n", this->name, trans);
  this->transparency = trans;
}
/**
   \brief Sets the Material Transparency.
   \param trans stes the Transparency from char*.
*/
void Material::setTransparency (char* trans)
{
  this->setTransparency (atof(trans));
}

/**
   \brief Adds a Texture Path to the List of already existing Paths
   \param pathName The Path to add.
*/
void Material::addTexturePath(const char* pathName)
{
  ResourceManager::getInstance()->addImageDir(pathName);
}

// MAPPING //

/**
   \brief Sets the Materials Diffuse Map
   \param dMap the Name of the Image to Use
*/
void Material::setDiffuseMap(const char* dMap)
{
  PRINTF(4)("setting Diffuse Map %s\n", dMap);
  if (this->diffuseTexture)
    ResourceManager::getInstance()->unload(this->diffuseTexture);

  //! \todo check if RESOURCE MANAGER is availiable
  //! \todo Textures from .mtl-file need special care.
  this->diffuseTexture = (Texture*)ResourceManager::getInstance()->load(dMap, IMAGE);
}

/**
   \brief Sets the Materials Ambient Map
   \param aMap the Name of the Image to Use
   \todo implement this
*/
void Material::setAmbientMap(const char* aMap)
{
  SDL_Surface* ambientMap;

}

/**
   \brief Sets the Materials Specular Map
   \param sMap the Name of the Image to Use
   \todo implement this
*/
void Material::setSpecularMap(const char* sMap)
{
  SDL_Surface* specularMap;

}

/**
   \brief Sets the Materials Bumpiness
   \param bump the Name of the Image to Use
   \todo implemet this
*/
void Material::setBump(const char* bump)
{

}
