/*!
  \file list.h
  \brief a File that includes a List-template
*/

#ifndef _LIST_H
#define _LIST_H

#include "compiler.h"
#ifndef NULL
#define NULL 0
#endif



//! a list element of the tList, 
template<class T> struct listElement
{
  listElement*        prev;                          //!< pointer to the previous listElement in the list
  T*                  curr;                          //!< pointer to the list payload/container
  listElement*        next;                          //!< pointer to the next listElement
};

/**
   \brief an iterator class

   this enables the user to iterate through a list very easely
*/
template<class T> class tIterator
{
 public:
  tIterator(listElement<T>* startElement);
  ~tIterator();
  
  T* nextElement();

 private:
  listElement<T>*    currentEl;                      //!< pointer to the current list element in the iterator
  listElement<T>*    tmpEl;                          //!< temp listElemnt pointer
};


/**
   \brief iterator constructor
   \param startElement:  the first list element from the tList

   normaly you will use it like this: 

   tIterator<char>* nameIterator = nameList->getIterator();
   char name* = nameIterator->nextElement(); 
   while( name != NULL)
   {
     PRINTF(3)("found name: %s in list\n", name);
     name = nameIterator->nextElement(); 
   }
   delete nameIterator;       
*/
template<class T>
inline tIterator<T>::tIterator (listElement<T>* startElement) 
{
  this->currentEl = startElement;
  this->tmpEl = NULL;
}


/**
   \brief the destructor
*/
template<class T>
inline tIterator<T>::~tIterator ()
{
  this->currentEl = NULL;
}


/**
   \brief use it to iterate through the list
   \returns next list element
*/
template<class T>
inline T* tIterator<T>::nextElement ()
{
  if( this->currentEl == NULL)
    return NULL;

  this->tmpEl = this->currentEl;
  this->currentEl = this->currentEl->next;
  return this->tmpEl->curr;
}



/**
   \brief the list template class

   you will use this as a generic list for all type of objects
*/
template<class T> class tList 
{
 public:
  tList ();
  ~tList ();

  void add(T* entity);
  void remove(T* entity);
  void flush();
  T* firstElement();
  T* lastElement();
  bool isEmpty();
  int getSize();
  //T* enumerate();
  tIterator<T>* getIterator();
  T* nextElement();
  T* nextElement(T* toEntity);
  T* toArray();
  void debug();

 private:
  unsigned int       size;             //!< the size (lenght) of the list
  listElement<T>*    first;            //!< pointer to the first element
  listElement<T>*    last;             //!< pointer to the last element
  listElement<T>*    currentEl;        //!< pointer to the current element
};


/**
   \brief the constructor
*/
template<class T>
inline tList<T>::tList () 
{
  this->first = NULL;
  this->last = NULL;
  this->size = 0;
}


/**
   \brief the deconstructor

   this will delete only the list. ATTENTION: the list is deleted, but the objects in the list will 
   not be deleted
*/
template<class T>
inline tList<T>::~tList () 
{
  this->currentEl = this->first;
  while(this->currentEl != NULL)
    {
      listElement<T>* le = this->currentEl->next;
      //delete this->currentEl->curr;
      delete this->currentEl;
      this->currentEl = le;
    }
  this->first = NULL;
  this->last = NULL;
  this->size = 0;
}


/**
   \brief add an entity to the list
   \param entity: the entity to add
*/
template<class T>
inline void tList<T>::add(T* entity)
{
  if( unlikely(entity == NULL)) return;
  listElement<T>* el = new listElement<T>;
  el->prev = this->last;
  el->curr = entity;
  el->next = NULL;

  this->last = el;

  if( unlikely(el->prev == NULL)) this->first = el; /* if first element */
  else el->prev->next = el;
  this->size++;
}


/**
   \brief remove an entity from the list
   \param entity: the entity to be removed
*/
template<class T>
inline void tList<T>::remove(T* entity)
{
  this->currentEl = this->first;
  listElement<T>* te;
  while( this->currentEl != NULL)
    {
      if( unlikely(this->currentEl->curr == entity))
	{ 
	  if( unlikely(this->currentEl->prev  == NULL)) this->first = this->currentEl->next;
	  else this->currentEl->prev->next = this->currentEl->next;

	  if( unlikely(this->currentEl->next == NULL)) this->last = this->currentEl->prev;
	  else this->currentEl->next->prev = this->currentEl->prev;

	  delete this->currentEl;
	  this->size--;
	  return;
	}
      this->currentEl = this->currentEl->next;
    }
}


/**
   \brief this will deletes the objects from the list
*/
template<class T>
inline void tList<T>::flush()
{
  this->currentEl = this->first;
  while(this->currentEl != NULL)
    {
      listElement<T>* le = this->currentEl->next;
      delete this->currentEl->curr;
      delete this->currentEl;
      this->currentEl = le;
    }
  this->first = NULL;
  this->last = NULL;
  this->size = 0;
}


/**
   \brief returns the first element of the list
   \returns first element
*/
template<class T>
inline T* tList<T>::firstElement()
{
  return this->first->curr;
}


/**
   \brief function returns the last element of the list
   \returns the last element
*/
template<class T>
inline T* tList<T>::lastElement()
{
  return this->last->curr;
}


/**
   \brief returns true if the list is empty
   \returns true if the list is empty
*/
template<class T>
inline bool tList<T>::isEmpty()
{
  return (this->size==0)?true:false;
}


/**
   \brief this returns the number of elements in the list
   \returns number of elements
*/
template<class T>
inline int tList<T>::getSize()
{
  return this->size;
}


/**
   \brief creates an itereator object and returns it
   \returns the iterator object to this list

   You will use this, if you want to iterate through the list
*/
template<class T>
inline tIterator<T>* tList<T>::getIterator()
{
  tIterator<T>* iterator = new tIterator<T>(this->first);
  return iterator;
}



/**
   \brief this returns the next element after toEntity or the first if toEntity is last
   \param toEntity: the entity after which is an entity, that has to be returned, sorry, terrible phrase
   \returns the element after toEntity
*/
template<class T>
inline T* tList<T>::nextElement(T* toEntity)
{
  //if( this->last == this->first == NULL) return NULL;
  if(this->size == 0) return NULL;
  if( toEntity == NULL) return this->first->curr;
  if( toEntity == this->last->curr ) return this->first->curr;
  this->currentEl = this->first;
  while(this->currentEl->curr != toEntity && this->currentEl->next != NULL)
    {
      this->currentEl = this->currentEl->next;
    }
  if(this->currentEl == NULL) return NULL;
  return this->currentEl->next->curr;
}


/**
   \brief creates an array out of the list (ATTENTION: not implemented)
   \returns pointer to the array beginning

   ATTENTION: function is not implemented and wont do anything
*/
template<class T>
T* tList<T>::toArray()
{
  return NULL;
}

#endif /* _LIST_H */
