/*! 
    \file particle_emitter.h
    \brief Definition of a ParticleEmitter
*/

#ifndef _PARTICLE_EMITTER_H
#define _PARTICLE_EMITTER_H

#include "p_node.h"

// FORWARD DEFINITION 
class ParticleSystem;
class TiXmlElement;

//! The form of the Emitter to emit from
typedef enum EMITTER_TYPE { EMITTER_DOT   = 1,
			    EMITTER_PLANE = 2,
			    EMITTER_SPHERE= 4,
			    EMITTER_CUBE  = 8 };

//! A class to handle an Emitter.
class ParticleEmitter : public PNode {

 public:
  ParticleEmitter(const Vector& direction, float angle = .5,
		  float emissionRate = 1.0, float velocity = 1.0);
  ParticleEmitter(const TiXmlElement* root);
  virtual ~ParticleEmitter(void);
  
  void loadParams(const TiXmlElement* root);

  /* controlling the emitter: interface */
  void start();
  void stop();
  void tick(float dt, ParticleSystem* system);

  /* controlling the behavour: these can be used as Animation interfaces */
  void setType(EMITTER_TYPE type);
  void setType(const char* type);
  void setSize(float emitterSize);
  void setEmissionRate(float emissionRate);
  void setInheritSpeed(float value);
  void setSpread(float angle, float randomAngle = 0.0);
  void setEmissionVelocity(float velocity, float randomVelocity = 0.0);

  /** \returns the type of the emitter */
  inline EMITTER_TYPE getType(void) const { return this->type; };
  /** \returns the Size of the emitter */
  inline float getSize(void) const { return this->emitterSize; };
  /** \returns the emissionRate */
  inline float getEmissionRate(void) const { return this->emissionRate; };
  /** \returns the inherit-speed-factor */
  inline float getInheritSpeed(void) const { return this->inheritSpeed; };
  /** \returns the SpreadAngle of the emitter */
  inline float getSpread(void) { return this->angle; };
  /** \returns the EmissionVelocity of the emitter */
  inline float getEmissionVelocity(void) { return this->velocity; };

  void debug(void);


 private:
  EMITTER_TYPE    type;              //!< The type of emitter this is
  float           emitterSize;       //!< The size of the emitter (not for EMITTER_DOT)
  float           inheritSpeed;      //!< How much speed the particle inherits from the Emitters speed \todo move this to the emitter
  Vector          direction;         //!< emition direction
  float           angle;             //!< max angle from the direction of the emitter
  float           randomAngle;       //!< random emission angle (angle +- angleRandom is the emitted angle.
  float           emissionRate;      //!< amount of particles per seconds emitted by emitter.
  float           velocity;          //!< the initial speed of a Particles.
  float           randomVelocity;    //!< the random variation from the initial Speed.

  float           saveTime;          //!< The time that was missing by the last Tick (otherwise there would be no emission when framefate is too big).
};

#endif /* _PARTICLE_EMITTER_H */
