
/*! 
    \file curve.h
    \brief A basic 3D curve framework
    
    Contains classes to handle curves
*/ 

#ifndef _CURVE_H
#define _CURVE_H

#include "vector.h"

//! An Enumerator that defines what sort of Curves are availible
enum CurveType {CURVE_BEZIER};


//! An abstract class to handle curves. Needed for the Tracking system in orxonox.
class Curve
{
 protected:
  //! Handles the curve-points (dynamic List)
  struct PathNode
  {
    int          number;       //!< The N-th node of this curve.
    float        factor;       //!< Curve specific multiplier factor.
    Vector       vFactor;      //!< A Vector-factor for multipliing.
    Vector       position;     //!< Vector Pointung to this curve-point.
    PathNode*    next;         //!< Pointer to the next Node.
  };

 public:
  Curve(void);

  void addNode(const Vector& newNode);
  void addNode(const Vector& newNode, unsigned int insertPosition);
  Vector getNode(unsigned int nodeToFind);
  /** \returns the count of nodes in this curve */
  inline int getNodeCount(void) const { return this->nodeCount; };
  /** \returns the directional Curve */
  Curve* getDirCurve(void) const { return this->dirCurve; };

  /** \param t the value on the curve [0-1] \returns Vector to the position */
  virtual Vector calcPos(float t) = 0;
  /** \param t the value on the curve [0-1] \returns the direction */
  virtual Vector calcDir(float t) = 0;
  /** \param t the value on the curve [0-1] \returns the acceleration */
  virtual Vector calcAcc(float t) = 0;
  /** \param t the value on the curve [0-1] \returns quaternion of the rotation */
  virtual Quaternion calcQuat(float t) = 0;
 
  // DEBUG
  void debug(void);

 private:
  /** \brief rebuilds the curve */
  virtual void rebuild(void) = 0;

 protected:
  int         nodeCount;       //!< The count of nodes the Curve has.
  Vector      curvePoint;      //!< The point on the Cureve at a local Time.
  float       localTime;       //!< If the time of one point is asked more than once the programm will not calculate it again.
  int         derivation;      //!< Which derivation of a Curve is this.

  Curve*      dirCurve;        //!< The derivation-curve of this Curve.

  PathNode*   firstNode;       //!< First node of the curve.
  PathNode*   currentNode;     //!< The node we are working with (the Last node).

};

//!    Class to handle bezier curves in 3-dimesnsional space
/**
   This Curve is good, for Fast Interaction. If you want to change it during the game, go on.
   !!be aware!! BezierCurves only flow through their first and last Node. Their Tangents at those Points a directed to the second and second-last Point.
*/
class BezierCurve : public Curve
{
 public:
  BezierCurve(void);
  BezierCurve(int derivation);
  virtual ~BezierCurve(void);

  virtual Vector calcPos(float t);
  virtual Vector calcDir(float t);
  virtual Vector calcAcc(float t);
  virtual Quaternion calcQuat(float t);
  
  
  Vector getPos(void) const;

 private:
  void rebuild(void);
};


//! B-Spline
/**
   class to handle b-spline in 3d space
*/
class BSplieCurve : public Curve
{


};

#endif /* _CURVE_H */
