/*! 
    \file resource_manager.h
    \brief The Resource Manager checks if a file/resource is loaded.

    If a file/resource was already loaded the resourceManager will 
    return a void pointer to the desired resource.
    Otherwise it will instruct the coresponding resource-loader to load,
    and receive a pointer to it.
*/

#ifndef _RESOURCE_MANAGER_H
#define _RESOURCE_MANAGER_H

#include "base_object.h"

#include "stdlibincl.h"

// FORWARD DEFINITION 
template<class T> class tList;

//! An eumerator for different fileTypes the resourceManager supports
typedef enum ResourceType { OBJ,
			    PRIM,
			    MD2,
			    WAV,
			    MP3,
			    OGG,
			    TTF,
			    IMAGE };

//! An enumerator for different UNLOAD-types. 
/**
   RP_NO:        will be unloaded on request
   RP_LEVEL:     will be unloaded at the end of a Level
   RP_CAMPAIGN:  will be unloaded at the end of a Campaign
   RP_GAME:      will be unloaded at the end of the whole Game (when closing orxonox)
*/
typedef enum ResourcePriority { RP_NO        =   0,
				RP_LEVEL     =   1,
				RP_CAMPAIGN  =   2,
				RP_GAME      =   4 };

//! A Struct that keeps track about A resource its name its Type, and so on
struct Resource
{
  void* pointer;             //!< Pointer to the Resource.
  int count;                 //!< How many times this Resource has been loaded.
  
  char* name;                //!< Name of the Resource.
  ResourceType type;         //!< ResourceType of this Resource.
  ResourcePriority prio;     //!< The Priority of this resource. (This will only be increased)

  // more specific
  float modelSize;
  unsigned int ttfSize;
  unsigned char ttfColorR;
  unsigned char ttfColorG;
  unsigned char ttfColorB;
  char* skinFileName;
};


//! The ResourceManager is a class, that decides if a file/resource should be loaded
/**
   If a file/resource was already loaded the resourceManager will 
   return a void pointer to the desired resource.
   Otherwise it will instruct the corresponding resource-loader to load,
   and receive the pointer to it.

   It does it by looking, if a desired file has already been loaded.
*/
class ResourceManager : public BaseObject 
{
 public:
  static ResourceManager* getInstance();
  virtual ~ResourceManager();

  bool setDataDir(const char* dataDir);
  /** \returns the Name of the data directory */
  inline const char* getDataDir(void) const {return this->dataDir;}

  bool checkDataDir(const char* fileInside);
  bool addImageDir(const char* imageDir);
  void* load(const char* fileName, ResourcePriority prio = RP_NO,
	     void* param1 = NULL, void* param2 = NULL, void* param3 = NULL);
  void* load(const char* fileName, ResourceType type, ResourcePriority prio = RP_NO,
	     void* param1 = NULL, void* param2 = NULL, void* param3 = NULL);
  bool unload(void* pointer, ResourcePriority prio = RP_NO);
  bool unload(Resource* resource, ResourcePriority = RP_NO);
  bool unloadAllByPriority(ResourcePriority prio);

  void debug(void);

  // utility functions of this class
  static bool isDir(const char* directory);
  static bool isFile(const char* fileName);
  static bool touchFile(const char* fileName);
  static bool deleteFile(const char* fileName);
  static char* homeDirCheck(const char* fileName);
  static char* getFullName(const char* fileName);

 private:
  ResourceManager();
  static ResourceManager* singletonRef;

  tList<Resource>* resourceList;       //!< The List of Resources, that has already been loaded.
  char* dataDir;                       //!< The Data Directory, where all relevant Data is stored.
  tList<char>* imageDirs;              //!< A list of directories in which images are stored.


  Resource* locateResourceByInfo(const char* fileName, ResourceType type, void* param1, void* param2, void* param3);
  Resource* locateResourceByPointer(const void* pointer);
  
};

#endif /* _RESOURCE_MANAGER_H */
