

/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD

#include "world.h"

#include "orxonox.h"

#include "state.h"

#include "p_node.h"
#include "null_parent.h"
#include "helper_parent.h"
#include "pilot_node.h"
#include "track_node.h"
#include "world_entity.h"
#include "player.h"
#include "camera.h"
#include "environment.h"
#include "skysphere.h"
#include "skybox.h"
#include "satellite.h"
#include "test_entity.h"
#include "terrain.h"
#include "light.h"
#include "text_engine.h"

#include "track_manager.h"
#include "garbage_collector.h"
#include "object_manager.h"
#include "animation_player.h"
#include "particle_engine.h"
#include "graphics_engine.h"
#include "physics_engine.h"
#include "fields.h"

#include "command_node.h"
#include "glmenu_imagescreen.h"
#include "list.h"
#include "game_loader.h"

#include "animation3d.h"

#include "substring.h"

#include "factory.h"

#include "projectile.h"

using namespace std;

WorldInterface* WorldInterface::singletonRef = 0;


/**
   \brief private constructor because of singleton
*/
WorldInterface::WorldInterface()
{
  this->worldIsInitialized = false;
  this->worldReference = NULL;
}

/**
   \brief public deconstructor
*/
WorldInterface::~WorldInterface()
{
  this->singletonRef = NULL;
  this->worldIsInitialized = false;
  this->worldReference = NULL;
}

/**
   \brief gets the singleton instance
   \returns singleton instance
*/
WorldInterface* WorldInterface::getInstance()
{
  if( singletonRef == NULL)
    singletonRef = new WorldInterface();
  return singletonRef;
}


/**
   \brief initializes the interface
   \param reference to the world

   if the worldinterface is not initilizes, there wont be any 
   useable interface
*/
void WorldInterface::init(World* world)
{
  this->worldReference = world;
  if( world != NULL)
    {
      this->worldIsInitialized = true;
      PRINTF(3)("WorldInterface up and running\n");
    }
}


/**
   \brief gets the entity list from the world
   \return entity list
*/
tList<WorldEntity>* WorldInterface::getEntityList()
{
  if( this->worldIsInitialized)
    return this->worldReference->getEntities();
  PRINT(1)("Someone tried to use the WorldInterface before it has been initizlized! this can result in SEGFAULTs!\n");
  return NULL;
}

CREATE_FACTORY(World);

World::World(const TiXmlElement* root)
{
  this->constuctorInit("", -1);
  this->path = NULL;
  
  this->loadParams(root);
}

/** 
    \brief create a new World
    
    This creates a new empty world!
*/
World::World (char* name)
{
  this->path = NULL;
  this->constuctorInit(name, -1);
  //NullParent* np = NullParent::getInstance();
}

/**
   \brief creates a new World...
   \param worldID with this ID
*/
World::World (int worldID)
{
  this->path = NULL;
  this->constuctorInit(NULL, worldID);
}

/** 
    \brief remove the World from memory
    
    delete everything explicitly, that isn't contained in the parenting tree!
    things contained in the tree are deleted automaticaly
*/
World::~World ()
{
  PRINTF(3)("World::~World() - deleting current world\n");
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->unbind(this->localPlayer);
  cn->reset();

  delete WorldInterface::getInstance();
  delete this->nullParent;
  delete this->entities;
  delete this->lightMan;
  delete this->trackManager;
  delete this->particleEngine;
  TextEngine::getInstance()->flush();
  delete AnimationPlayer::getInstance(); // this should be at the end of the unloading sequence.
  delete PhysicsEngine::getInstance();
  //delete garbagecollecor
  //delete animator

  LoadClassDescription::printAll();

  ResourceManager::getInstance()->unloadAllByPriority(RP_LEVEL);
}

/**
   \brief initializes the world. 

   set all stuff here that is world generic and does not use to much memory
   because the real init() function StoryEntity::init() will be called
   shortly before start of the game.  
   since all worlds are initiated/referenced before they will be started.
   NO LEVEL LOADING HERE - NEVER!
*/
void World::constuctorInit(char* name, int worldID)
{
  this->setClassID(CL_WORLD, "World");

  //this->worldName = name;
  //this->worldName = new char[strlen(name)+1];
  //strcpy(this->worldName, name);
  this->debugWorldNr = worldID;
  this->entities = new tList<WorldEntity>();
}

void World::loadParams(const TiXmlElement* root)
{
  const char *string;
  char *name;
  int id;

  PRINTF0("Creating a World\n");

  LoadParam<World>(root, "identifier", this, &World::setStoryID)
    .describe("Sets the StoryID of this world");
  LoadParam<World>(root, "nextid", this, &World::setNextStoryID)
    .describe("Sets the ID of the next world");
  LoadParam<World>(root, "path", this, &World::setPath)
    .describe("The Filename of this World (relative from the data-dir)");


  /*  
  // identifier
  string = grabParameter( root, "identifier");
  if( string == NULL || sscanf(string, "%d", &id) != 1)
  {
  PRINTF0("World is missing a proper 'identifier'\n");
  this->setStoryID( -1);
  }
  else setStoryID( id);

  // next id
  string = grabParameter( root, "nextid");
  if( string == NULL || sscanf(string, "%d", &id) != 1)
  {
  PRINTF0("World is missing a proper 'nextid'\n");
  this->setStoryID( -1);
  }
  else setNextStoryID( id);
  

  // path
  string = grabParameter( root, "path");
  if( string == NULL)
  {
  PRINTF0("World is missing a proper 'path'\n");
  this->setPath( NULL);
  }
  else
  {
  name = new char[strlen(string + 2)];
  strcpy( name, string);
  this->setPath( name);
  }
  */
}


/**
   \brief this is executed before load

   since the load function sometimes needs data, that has been init before
   the load and after the proceeding storyentity has finished
*/
ErrorMessage World::preLoad()
{
  /* init the world interface */
  WorldInterface* wi = WorldInterface::getInstance();
  wi->init(this);
  this->garbageCollector = GarbageCollector::getInstance();

  this->particleEngine = ParticleEngine::getInstance();
  this->trackManager = TrackManager::getInstance();
  this->lightMan = LightManager::getInstance();
  this->nullParent = NullParent::getInstance ();
  this->nullParent->setName ("NullParent");

  AnimationPlayer::getInstance(); // initializes the animationPlayer
  PhysicsEngine::getInstance();

  this->localCamera = new Camera();
  this->localCamera->setName ("Camera");
  
  State::getInstance()->setCamera(this->localCamera, this->localCamera->getTarget());

  GraphicsEngine::getInstance()->displayFPS(true);
}


/**
   \brief loads the World by initializing all resources, and set their default values.
*/
ErrorMessage World::load()
{	
  PRINTF(3)("> Loading world: '%s'\n", getPath());
  TiXmlElement* element;
  GameLoader* loader = GameLoader::getInstance();
  
  if( getPath() == NULL)
    {
      PRINTF(1)("World has no path specified for loading");
      this->loadDebugWorld(this->getStoryID());
      return (ErrorMessage){213,"Path not specified","World::load()"};
    }
  
  TiXmlDocument* XMLDoc = new TiXmlDocument( path);
  // load the campaign document
  if( !XMLDoc->LoadFile())  
  {
    // report an error
    PRINTF(1)("loading XML File: %s @ %d:%d\n", XMLDoc->ErrorDesc(), XMLDoc->ErrorRow(), XMLDoc->ErrorCol());
    delete XMLDoc;
    return (ErrorMessage){213,"XML File parsing error","World::load()"};
  }
  
  // check basic validity
  TiXmlElement* root = XMLDoc->RootElement();
  assert( root != NULL);
  
  if( root == NULL || root->Value() == NULL || strcmp( root->Value(), "WorldDataFile"))
    {
      // report an error
      PRINTF(1)("Specified XML File is not an orxonox world data file (WorldDataFile element missing)\n");
      delete XMLDoc;
      return (ErrorMessage){213,"Path not a WorldDataFile","World::load()"};
    }
  
  // load the parameters
  // name
  char* temp;
  const char* string = grabParameter( root, "name");
  if( string == NULL)
    {
      PRINTF(2)("World is missing a proper 'name'\n");
      string = "Unknown";
      temp = new char[strlen(string + 2)];
      strcpy( temp, string);
      this->worldName = temp;
    }
  else
    {
      temp = new char[strlen(string + 2)];
      strcpy( temp, string);
      this->worldName = temp;
    }
  ////////////////
  // LOADSCREEN //
  ////////////////
  element = root->FirstChildElement("LoadScreen");
  if (element == NULL)
    {
      PRINTF(2)("no LoadScreen specified, loading default\n");

      glmis->setBackgroundImage("pictures/load_screen.jpg");
      this->glmis->setMaximum(8);
      this->glmis->draw();
    }
  else
    {
      this->glmis->loadParams(element);
      this->glmis->draw();
    }
  this->glmis->draw();
  // find WorldEntities
  element = root->FirstChildElement("WorldEntities");
  
  if( element == NULL)
    {
      PRINTF(1)("World is missing 'WorldEntities'\n");
    }
  else
    {
      element = element->FirstChildElement();
      // load Players/Objects/Whatever
      PRINTF(4)("Loading WorldEntities\n");
      while( element != NULL)
	{
	  WorldEntity* created = dynamic_cast<WorldEntity*>( loader->fabricate( element));
	  if( created != NULL) this->spawn( created);
	  // if we load a 'Player' we use it as localPlayer
	  //todo do this more elegant 
	  if( element->Value() != NULL && !strcmp( element->Value(), "Player")) localPlayer = (Player*) created;
	  if( element->Value() != NULL && !strcmp( element->Value(), "SkyBox")) sky = (SkyBox*) created;
	  element = element->NextSiblingElement();
	  glmis->step(); //! \todo temporary
	}
      PRINTF(4)("Done loading WorldEntities\n");
    }
  
  // find Track
  element = root->FirstChildElement( "Track");
  if( element == NULL)
    {
      PRINTF(0)("World is missing a 'Track'\n");
    }
  else
    {	
      //load track
      PRINTF(4)("Loading Track\n");

      trackManager->load( element);
      trackManager->finalize();
    }
  
  // free the XML data

  delete XMLDoc;
  /* GENERIC LOADING PROCESS FINISHED */
  
  // bind input
  Orxonox *orx = Orxonox::getInstance ();
  orx->getLocalInput()->bind (localPlayer);
  
  // bind camera
  //this->localCamera->bind (localPlayer);
  this->localPlayer->addChild (this->localCamera);
      

  lightMan->setAmbientColor(.1,.1,.1);
  lightMan->addLight();
  //      lightMan->setAttenuation(1.0, .01, 0.0);
  //      lightMan->setDiffuseColor(1,1,1);
  //  lightMan->addLight(1);
  //  lightMan->setPosition(20, 10, -20);
  //  lightMan->setDiffuseColor(0,0,0);
  //lightMan->debug();
  lightMan->setPosition(-5.0, 10.0, -40.0);
  
  //	    trackManager->setBindSlave(env);
  PNode* tn = trackManager->getTrackNode();
  tn->addChild(this->localPlayer);
  
  //localCamera->setParent(TrackNode::getInstance());
  tn->addChild(this->localCamera);
  localCamera->lookAt(tn);
  localCamera->setMode(PNODE_MOVEMENT);
  this->localPlayer->setMode(PNODE_ALL);
  Vector* cameraOffset = new Vector (0, 5, -10);
  trackManager->condition(2, LEFTRIGHT, this->localPlayer);
  
  this->sky->setParent(this->localCamera);

  // initialize debug coord system
  objectList = glGenLists(1);
  glNewList (objectList, GL_COMPILE);
  
  //trackManager->drawGraph(.01);
  //trackManager->debug(2);
  glEndList();

  terrain = new Terrain("worlds/newGround.obj");
  terrain->setRelCoor(Vector(0,-10,0));
  this->spawn(terrain);



  // Creating a Test Particle System
  ParticleSystem* system = new ParticleSystem(100000, PARTICLE_SPRITE);
  system->setLifeSpan(1);
  system->setConserve(.8);
  system->setRadius(4, 3, 1, 2);
  system->setColor(.5,0,0,.5, 1,1,0,1, 0,0,0,0);

  // Creating a Test Particle Emitter
  ParticleEmitter* emitter = new ParticleEmitter(Vector(-1, 0, 0), M_PI_4, 400, .5);
  emitter->setType(EMITTER_DOT);
  emitter->setSize(20);
  emitter->setParent(this->localPlayer);
  emitter->setRelCoor(Vector(-3,0,0));

  Field* gravity = new Gravity();
  gravity->setMagnitude(10.0);
  //  gravity->setParent(this->localCamera->getTarget());
  
  // Add the Flow from the Emitter into the System
  particleEngine->addConnection(emitter, system);

  new PhysicsConnection(system, gravity);
  //    new PhysicsConnection(this->localPlayer, gravity);
  

  TestEntity* testEntity = new TestEntity();
  testEntity->setRelCoor(Vector(570, 10, -15));
  //testEntity->setRelCoor(Vector(25, -10, -20));
  testEntity->setRelDir(Quaternion(M_PI, Vector(0, 1, 0)));
  this->spawn(testEntity);
  //  this->localPlayer->addChild(testEntity);

  new PhysicsConnection(testEntity, gravity);
}



/**
   \brief loads the debug world: only for experimental stuff
*/
void World::loadDebugWorld(int worldID)
{
  printf("ajsdflkajs;dlfaj;slfja;lsjf;lajsf;la;sdfkja;lskdjfashdfklajshdflkjasdfh\n");
  /*monitor progress*/
  this->glmis->step();
  // stuff beyond this point remains to be loaded properly

  // initializing the TrackManager
  this->trackManager = TrackManager::getInstance();
  //trackManager->addPoint(Vector(0,0,0));
  trackManager->addPoint(Vector(150, -35, 5));
  trackManager->addPoint(Vector(200,-35, 5));
  trackManager->addPoint(Vector(250, -35, 5));
  trackManager->addPoint(Vector(320,-33,-.55));
  trackManager->setDuration(1);
  trackManager->setSavePoint();

  trackManager->addPoint(Vector(410, 0, 0));
  trackManager->addPoint(Vector(510, 20, -10));
  trackManager->addPoint(Vector(550, 20, -10));
  trackManager->addPoint(Vector(570, 20, -10));
  trackManager->setDuration(2);
      
  trackManager->forkS("testFork1,testFork2");
  trackManager->workOn("testFork1");
  trackManager->addPoint(Vector(640, 25, -30));
  trackManager->addPoint(Vector(700, 40, -120));
  trackManager->addPoint(Vector(800, 50, -150));
  trackManager->addPoint(Vector(900, 60, -100));
  trackManager->addPoint(Vector(900, 60, -70));
  trackManager->addPoint(Vector(990, 65, -15));
  trackManager->addPoint(Vector(1050, 65, -10));
  trackManager->addPoint(Vector(1100, 65, -20));
  trackManager->setDuration(4);

  trackManager->workOn("testFork2");
  trackManager->addPoint(Vector(640, 25, 20));
  trackManager->addPoint(Vector(670, 50, 120));
  trackManager->addPoint(Vector(700, 70, 80));
  trackManager->addPoint(Vector(800, 70, 65));
  trackManager->addPoint(Vector(850, 65, 65));
  trackManager->addPoint(Vector(920, 35, 40));
  trackManager->addPoint(Vector(945, 40, 40));
  trackManager->addPoint(Vector(970, 24, 40));
  trackManager->addPoint(Vector(1000, 40, -7));
  trackManager->setDuration(4);
      
      
  trackManager->joinS("testFork1,testFork2");
	
  trackManager->addPoint(Vector(1200, 60, -50));
  trackManager->addPoint(Vector(1300, 50, -50));
  trackManager->addPoint(Vector(1400, 40, -50));
  trackManager->addPoint(Vector(1500, 40, -60));
  trackManager->addPoint(Vector(1600, 35, -55));
  trackManager->addPoint(Vector(1700, 45, -40));
  trackManager->addPoint(Vector(1750, 60, -40));
  trackManager->addPoint(Vector(1770, 80, -40));
  trackManager->addPoint(Vector(1800, 100, -40));
  trackManager->setDuration(10);
  
  trackManager->finalize();
  

  // LIGHT initialisation
  lightMan->setAmbientColor(.1,.1,.1);
  lightMan->addLight();
  lightMan->debug();

  switch(this->debugWorldNr)
    {
      /*
	this loads the hard-coded debug world. this only for simplicity and will be 
	removed by a reald world-loader, which interprets a world-file.
	if you want to add an own debug world, just add a case DEBUG_WORLD_[nr] and
	make whatever you want...
      */
    case DEBUG_WORLD_0:
      {
	lightMan->setPosition(-5.0, 10.0, -40.0);


	this->localPlayer = new Player ();
	this->localPlayer->setName ("player");
	this->spawn (this->localPlayer);
	this->localPlayer->setRelCoor(Vector(5,0,0));
	/*monitor progress*/
	this->glmis->step();

	Field* testField = new Gravity();
	testField->setMagnitude(10);
	new PhysicsConnection(this->localPlayer, testField);

	// bind camera
	this->localCamera = new Camera();
	this->localCamera->setName ("camera");
	/*monitor progress*/
	this->glmis->step();


	// Create SkySphere
	this->sky = new Skysphere("pictures/sky-replace.jpg");
	this->sky->setName("SkySphere");
	this->spawn(this->sky);
	this->localCamera->addChild(this->sky);
	this->sky->setMode(PNODE_MOVEMENT);
	/*monitor progress*/
	this->glmis->step();


	terrain = new Terrain("worlds/newGround.obj");
	terrain->setRelCoor(Vector(0,-10,0));
	this->spawn(terrain);
	/*monitor progress*/
	this->glmis->step();

	this->pilotNode = new PilotNode();
	this->pilotNode->addChild(this->localPlayer);
	this->pilotNode->addChild(this->localCamera);
	// bind input
	Orxonox *orx = Orxonox::getInstance ();
	//orx->getLocalInput()->bind (this->pilotNode);	    
	
	/*
	PNode* tn = trackManager->getTrackNode();
	tn->addChild(this->localPlayer);
	this->localCamera->lookAt(tn);
	
	tn->addChild(this->localCamera);
	this->localPlayer->setMode(PNODE_ALL);
	trackManager->condition(2, LEFTRIGHT, this->localPlayer);
	*/
	this->glmis->step();
	break;
      }
    case DEBUG_WORLD_1:
      {

	break;
      }
    case DEBUG_WORLD_2:
      {

	break;
      }
    default:
      break;
    }
}



/**
   \brief initializes a new World shortly before start

   this is the function, that will be loaded shortly before the world is 
   started
*/
ErrorMessage World::init()
{
  this->bPause = false;
  this->pilotNode = NULL;
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->addToWorld(this);
  cn->enable(true);
}


/**
   \brief starts the World
*/
ErrorMessage World::start()
{
  PRINTF(3)("World::start() - starting current World: nr %i\n", this->debugWorldNr);
  this->bQuitOrxonox = false;
  this->bQuitCurrentGame = false;
  this->mainLoop();
}

/**
   \brief stops the world.

   This happens, when the player decides to end the Level.
*/
ErrorMessage World::stop()
{
  PRINTF(3)("World::stop() - got stop signal\n");
  this->bQuitCurrentGame = true;
}

/**
   \brief pauses the Game
*/
ErrorMessage World::pause()
{
  this->isPaused = true;
}

/**
   \brief ends the pause Phase
*/
ErrorMessage World::resume()
{
  this->isPaused = false;
}

/**
   \brief destroys the World
*/
ErrorMessage World::destroy()
{

}

/**
   \brief shows the loading screen
*/
void World::displayLoadScreen ()
{
  PRINTF(3)("World::displayLoadScreen - start\n"); 
  
  //GLMenuImageScreen* 
  this->glmis = new GLMenuImageScreen();
  this->glmis->init();
  this->glmis->setMaximum(8);
  //  this->glmis->draw();
 
  PRINTF(3)("World::displayLoadScreen - end\n"); 
}

/**
   \brief removes the loadscreen, and changes over to the game

   \todo take out the delay
*/
void World::releaseLoadScreen ()
{
  PRINTF(3)("World::releaseLoadScreen - start\n"); 
  this->glmis->setValue(this->glmis->getMaximum());
  PRINTF(3)("World::releaseLoadScreen - end\n"); 
  delete this->glmis;
}


/**
   \brief gets the list of entities from the world
   \returns entity list
*/
tList<WorldEntity>* World::getEntities()
{
  return this->entities;
}


/**
   \brief this returns the current game time
   \returns elapsed game time
*/
double World::getGameTime()
{
  return this->gameTime;
}


/** 
    \brief checks for collisions
    
    This method runs through all WorldEntities known to the world and checks for collisions 
    between them. In case of collisions the collide() method of the corresponding entities 
    is called.
*/
void World::collide ()
{
  /*
  List *a, *b;
  WorldEntity *aobj, *bobj;
   
  a = entities;
  
  while( a != NULL)
    {
      aobj = a->nextElement();
      if( aobj->bCollide && aobj->collisioncluster != NULL)
	{
	  b = a->nextElement();
	  while( b != NULL )
	    {
	      bobj = b->nextElement();
	      if( bobj->bCollide && bobj->collisioncluster != NULL )
		{
		  unsigned long ahitflg, bhitflg;
		  if( check_collision ( &aobj->place, aobj->collisioncluster, 
					&ahitflg, &bobj->place, bobj->collisioncluster, 
					&bhitflg) );
		  {
		    aobj->collide (bobj, ahitflg, bhitflg);
		    bobj->collide (aobj, bhitflg, ahitflg);
		  }
		}
	      b = b->nextElement();
	    }
	}
      a = a->enumerate();
    }
  */
}

/** 
    \brief runs through all entities calling their draw() methods
*/
void World::draw ()
{
  /* draw entities */
  WorldEntity* entity;
  glLoadIdentity();

  //entity = this->entities->enumerate();
  tIterator<WorldEntity>* iterator = this->entities->getIterator();
  entity = iterator->nextElement();
  while( entity != NULL ) 
    { 
      if( entity->bDraw ) entity->draw();
      //entity = this->entities->nextElement();
      entity = iterator->nextElement();
    }
  delete iterator;
  
  glCallList (objectList);

  TextEngine::getInstance()->draw();
  particleEngine->draw(); //!< \todo should be dts like in the Trunk;

  lightMan->draw(); // must be at the end of the drawing procedure, otherwise Light cannot be handled as PNodes //
}


/**
   \brief function to put your own debug stuff into it. it can display informations about
   the current class/procedure
*/
void World::debug()
{
  PRINTF(2)("debug() - starting debug\n");
  PNode* p1 = NullParent::getInstance ();
  PNode* p2 = new PNode (Vector(2, 2, 2), p1);
  PNode* p3 = new PNode (Vector(4, 4, 4), p1);
  PNode* p4 = new PNode (Vector(6, 6, 6), p2);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p1->shiftCoor (Vector(-1, -1, -1));

  printf("World::debug() - shift\n");
  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();
  
  p1->update (0);

  printf ("World::debug() - update\n");
  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p2->shiftCoor (Vector(-1, -1, -1));
  p1->update (0);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  p2->setAbsCoor (Vector(1,2,3));


 p1->update (0);

  p1->debug ();
  p2->debug ();
  p3->debug ();
  p4->debug ();

  delete p1;
  
  
  /*
  WorldEntity* entity;
  printf("counting all entities\n");
  printf("World::debug() - enumerate()\n");
  entity = entities->enumerate();  
  while( entity != NULL ) 
    { 
      if( entity->bDraw ) printf("got an entity\n");
      entity = entities->nextElement();
    }
  */
}


/**
  \brief main loop of the world: executing all world relevant function

  in this loop we synchronize (if networked), handle input events, give the heart-beat to
  all other member-entities of the world (tick to player, enemies etc.), checking for
  collisions drawing everything to the screen.
*/
void World::mainLoop()
{
  this->lastFrame = SDL_GetTicks ();
  PRINTF(3)("World::mainLoop() - Entering main loop\n");
  while( !this->bQuitOrxonox && !this->bQuitCurrentGame) /* \todo implement pause */
    {
      PRINTF(3)("World::mainloop() - number of entities: %i\n", this->entities->getSize());
      // Network
      this->synchronize ();
      // Process input
      this->handleInput ();
      if( this->bQuitCurrentGame || this->bQuitOrxonox)
	  break;
      // Process time
      this->tick ();
      // Update the state
      this->update ();     
      // Process collision
      this->collide ();
      // Draw
      this->display ();

      //      for( int i = 0; i < 5000000; i++) {}
      /* \todo this is to slow down the program for openGl Software emulator computers, reimplement*/
    }
  PRINTF(3)("World::mainLoop() - Exiting the main loop\n");
}


/**
   \brief synchronize local data with remote data
*/
void World::synchronize ()
{
  // Get remote input
  // Update synchronizables
}


/**
   \brief run all input processing

   the command node is the central input event dispatcher. the node uses the even-queue from
   sdl and has its own event-passing-queue.
*/
void World::handleInput ()
{
  // localinput
  CommandNode* cn = Orxonox::getInstance()->getLocalInput();
  cn->process();
  // remoteinput
}


/**
   \brief advance the timeline

   this calculates the time used to process one frame (with all input handling, drawing, etc)
   the time is mesured in ms and passed to all world-entities and other classes that need
   a heart-beat.
*/
void World::tick ()
{
  Uint32 currentFrame = SDL_GetTicks();
  if(!this->bPause)
    {
      this->dt = currentFrame - this->lastFrame;
      
      if( this->dt > 0)
	{
	  float fps = 1000/dt;

	  // temporary, only for showing how fast the text-engine is
	  char tmpChar[20];
	  sprintf(tmpChar, "fps: %4.0f", fps);
	}
      else
	{
	  /* the frame-rate is limited to 100 frames per second, all other things are for
	     nothing.
	  */
	  PRINTF(2)("fps = 1000 - frame rate is adjusted\n");
	  SDL_Delay(10);
	  this->dt = 10;
	}
      //this->timeSlice (dt);
      
      /* function to let all entities tick (iterate through list) */
      this->dtS = (float)this->dt / 1000.0;      
      this->gameTime += this->dtS;
      //entity = entities->enumerate(); 
      tIterator<WorldEntity>* iterator = this->entities->getIterator();
      WorldEntity* entity = iterator->nextElement();
      while( entity != NULL) 
	{ 
	  entity->tick (this->dtS);
	  entity = iterator->nextElement();
	}
      delete iterator;

      /* update tick the rest */
      this->trackManager->tick(this->dt);
      this->localCamera->tick(this->dt);
      AnimationPlayer::getInstance()->tick(this->dtS);
      PhysicsEngine::getInstance()->tick(this->dtS);


      particleEngine->tick(this->dtS);
      this->garbageCollector->tick(this->dtS);
	  
      /* actualy the Graphics Engine should tick the world not the other way around...
	 but since we like the things not too complicated we got it this way around
	 until there is need or time to do it the other way around.
	 \todo: GraphicsEngine ticks world: separation of processes and data...
      */
      GraphicsEngine::getInstance()->tick(this->dtS);
    }
  this->lastFrame = currentFrame;
}


/**
   \brief this function gives the world a consistant state

   after ticking (updating the world state) this will give a constistant
   state to the whole system.
*/
void World::update()
{
  this->garbageCollector->update();
  this->nullParent->update (this->dtS);
}


/**
   \brief render the current frame
   
   clear all buffers and draw the world
*/
void World::display ()
{
  // clear buffer
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  // set camera
  this->localCamera->apply ();
  // draw world
  this->draw();
  // draw HUD
  /* \todo draw HUD */
  // flip buffers
  SDL_GL_SwapBuffers();
  //SDL_Surface* screen = Orxonox::getInstance()->getScreen ();
  //SDL_Flip (screen);
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
*/
void World::spawn(WorldEntity* entity)
{
  this->entities->add (entity);
  entity->postSpawn ();
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
   \param absCoor At what coordinates to add this entity.
   \param absDir In which direction should it look.
*/
void World::spawn(WorldEntity* entity, Vector* absCoor, Quaternion* absDir)
{
  this->entities->add (entity);

  entity->setAbsCoor (*absCoor);
  entity->setAbsDir (*absDir);

  entity->postSpawn ();
}


/**
   \brief add and spawn a new entity to this world
   \param entity to be added
   \param entity to be added to (PNode)
   \param At what relative  coordinates to add this entity.
   \param In which relative direction should it look.
*/
void World::spawn(WorldEntity* entity, PNode* parentNode, 
		  Vector* relCoor, Quaternion* relDir, 
		  int parentingMode)
{
  this->nullParent = NullParent::getInstance();
  if( parentNode != NULL)
    {
      parentNode->addChild (entity);
      
      entity->setRelCoor (*relCoor);
      entity->setRelDir (*relDir);
      entity->setMode(parentingMode);
      
      this->entities->add (entity);
      
      entity->postSpawn ();
    }
}



/**
  \brief commands that the world must catch
  \returns false if not used by the world
*/
bool World::command(Command* cmd)
{
  if( !strcmp( cmd->cmd, CONFIG_NAME_VIEW0)) this->localCamera->setViewMode(VIEW_NORMAL);
  else if( !strcmp( cmd->cmd, CONFIG_NAME_VIEW1)) this->localCamera->setViewMode(VIEW_BEHIND);
  else if( !strcmp( cmd->cmd, CONFIG_NAME_VIEW2)) this->localCamera->setViewMode(VIEW_FRONT);
  else if( !strcmp( cmd->cmd, CONFIG_NAME_VIEW3)) this->localCamera->setViewMode(VIEW_LEFT);
  else if( !strcmp( cmd->cmd, CONFIG_NAME_VIEW4)) this->localCamera->setViewMode(VIEW_RIGHT);
  else if( !strcmp( cmd->cmd, CONFIG_NAME_VIEW5)) this->localCamera->setViewMode(VIEW_TOP);
  else if(this->pilotNode != NULL) if( !strcmp( cmd->cmd, "cursor")) this->pilotNode->command(cmd);
  return false;
}

void World::setPath( const char* name)
{
  if (this->path)
    delete this->path;
  if (ResourceManager::isFile(name))
  {
    this->path = new char[strlen(name)+1];
    strcpy(this->path, name);
  }
  else
    {
      this->path = new char[strlen(ResourceManager::getInstance()->getDataDir()) + strlen(name) +1];
      sprintf(this->path, "%s%s", ResourceManager::getInstance()->getDataDir(), name);
    }
}

const char* World::getPath( void)
{
  return path;
}
