/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: 

   This code is inspired by Christian Meyers CommandNode code (in Orxonox v0.2.3)
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_EVENT

#include "event_handler.h"
#include "event_listener.h"
#include "event.h"

#include "ini_parser.h"
#include "keynames.h"


using namespace std;


/**
   \brief standard constructor
*/
EventHandler::EventHandler () 
{
  this->setClassID(CL_EVENT_HANDLER, "EventHandler"); 

  this->listeners = new EventListener**[ES_NUMBER];
  for(int i = 0; i < ES_NUMBER; ++i)
    this->listeners[i] = new EventListener*[SDLK_LAST];

  /* now initialize them all to zero */
  for(int i = 0; i < ES_NUMBER; ++i)
    {
      for(int j = 0; j < SDLK_LAST; ++j)
	{
	  this->listeners[i][j] = NULL;
	}
    }
}

/**
   \brief the singleton reference to this class
*/
EventHandler* EventHandler::singletonRef = NULL;

/**
   \returns a Pointer to this Class
*/
EventHandler* EventHandler::getInstance(void)
{
  if (!EventHandler::singletonRef)
    EventHandler::singletonRef = new EventHandler();
  return EventHandler::singletonRef;
}

/**
   \brief standard deconstructor

*/
EventHandler::~EventHandler () 
{
  EventHandler::singletonRef = NULL;
}


/**
   \brief loads new key bindings from a file
   \param filename: The path and name of the file to load the bindings from
*/
void EventHandler::loadKeyBindings (const char* fileName)
{
  FILE* stream;
  
  PRINTF(4)("Loading key bindings from %s\n", fileName);
  
  // remove old bindings if present
  if( this->keyAliases != NULL)
    {
      free (this->keyAliases);
      this->keyAliases = NULL;
    }
  
  // create parser
  IniParser parser(fileName);
  if( parser.getSection (CONFIG_SECTION_PLAYER "1") == -1)
    {
      PRINTF(1)("Could not find key bindings " CONFIG_SECTION_PLAYER"1 in %s\n", fileName);
      return;
    }
  // allocate empty lookup table
  this->keyAliases = (KeyBindings*) calloc (1, sizeof (KeyBindings));
  
  char namebuf[256];
  char valuebuf[256];
  memset (namebuf, 0, 256);
  memset (valuebuf, 0, 256);
  int* index;
  
  while( parser.nextVar (namebuf, valuebuf) != -1)
    {
      //index = nameToIndex (valuebuf);
      int c;
      if( (c = keynameToSDLK (valuebuf)) != -1)
	{
	  index[1] = c; index[0] = 0;
	}
      if( (c = buttonnameToSDLB (valuebuf)) != -1)
	{
	  index[1] = c; index[0] = 1;
	}

      switch( index[0])
	{
	case 0:
	  PRINTF(4)("Key binding %d(%s) set to %s\n", index[1], SDLKToKeyname( index[1]), namebuf);
	  strcpy (this->keyAliases->keys[index[1]], namebuf);
	  break;
	case 1:
	  PRINTF(4)("Button binding %d(%s) set to %s\n", index[1], SDLBToButtonname( index[1]), namebuf);
	  strcpy (this->keyAliases->buttons[index[1]], namebuf);
	  break;
	default:
	  break;
	}
      memset (namebuf, 0, 256);
      memset (valuebuf, 0, 256);
    }


  // PARSE MISC SECTION
  if( parser.getSection (CONFIG_SECTION_MISC_KEYS) == -1)
    {
      PRINTF(1)("Could not find key bindings in %s\n", fileName);
      return;
    }

  while( parser.nextVar (namebuf, valuebuf) != -1)
    {
      //index = nameToIndex (valuebuf);     
      int c;
      if( (c = keynameToSDLK (valuebuf)) != -1)
	{
	  index[1] = c; index[0] = 0;
	}
      if( (c = buttonnameToSDLB (valuebuf)) != -1)
	{
	  index[1] = c; index[0] = 1;
	}


      switch( index[0])
	{
	case 0:
	  PRINTF(4)("Key binding %d(%s) set to %s\n", index[1], SDLKToKeyname( index[1]), namebuf);
	  strcpy (keyAliases->keys[index[1]], namebuf);
	  break;
	case 1:
	  PRINTF(4)("Button binding %d(%s) set to %s\n", index[1], SDLBToButtonname( index[1]), namebuf);
	  strcpy (keyAliases->buttons[index[1]], namebuf);
	  break;
	default:
	  break;
	}
      memset (namebuf, 0, 256);
      memset (valuebuf, 0, 256);
    }
}


void EventHandler::setState(elState state)
{
  this->state = state;
}

void EventHandler::subscribeListener(EventListener* el, elState state, int eventType)
{
  if( likely(this->listeners[state][eventType] == NULL))
    this->listeners[state][eventType] = el;
  else
    PRINTF(0)("Someone tried to subscribe to event %i @ state %i but this event has already been subscribed\n", eventType, state);
}


void EventHandler::unsubscribeListener(int eventType, elState state)
{
  this->listeners[state][eventType] = NULL;
}

void EventHandler::flush()
{
  for(int i = 0; i < ES_NUMBER; ++i)
    {
      for(int j = 0; j < SDLK_LAST; ++j)
	{
	  this->listeners[i][j] = NULL;
	}
    }
}



void EventHandler::process()
{
  SDL_Event event;
  //Command cmd;
  Event ev;
  EventListener* listener;
  while( SDL_PollEvent (&event))
    {
      //memset (ev.cmd, 0, CMD_LENGHT); 
      switch( event.type)
	{
	case SDL_KEYDOWN:
	  ev.bPressed = true;
	  ev.type = event.key.keysym.sym;
	  break;
	case SDL_KEYUP:
	  ev.bPressed = false;
	  ev.type = event.key.keysym.sym;
	  break;
	case SDL_MOUSEMOTION:
	  ev.bPressed = false;
	  ev.type = EV_MOUSE_MOTION;
	  ev.x = event.motion.x;
	  ev.y = event.motion.y;
	  ev.xRel = event.motion.xrel;
	  ev.yRel = event.motion.yrel;
	  break;
	case SDL_MOUSEBUTTONUP:
	  ev.bPressed = false;
	  ev.type = EV_MOUSE_BUTTON;
	  break;
	case SDL_MOUSEBUTTONDOWN:
	  ev.bPressed = true;
	  ev.type = EV_MOUSE_BUTTON;
	  break;
	case SDL_JOYAXISMOTION:
	  ev.bPressed = false;
	  ev.type = EV_JOY_AXIS_MOTION;
	  break;
	case SDL_JOYBALLMOTION:
	  ev.bPressed = false;
	  ev.type = EV_JOY_BALL_MOTION;
	  break;
	case SDL_JOYHATMOTION:
	  ev.bPressed = false;
	  ev.type = EV_JOY_HAT_MOTION;
	  break;
	case SDL_JOYBUTTONDOWN:
	  ev.bPressed = true;
	  ev.type = EV_JOY_BUTTON;
	  break;
	case SDL_JOYBUTTONUP:
	  ev.bPressed = true;
	  ev.type = EV_JOY_BUTTON;
	  break;
	default:
	  break;
	}

      listener = this->listeners[this->state][event.key.keysym.sym];
      //if( listener != NULL)
      //listener->process();
      
    }
}
