

/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: ...
*/

#include "camera.h"

#include "world.h"
#include "world_entity.h"
#include "vector.h"

using namespace std;

////////////
// CAMERA //
////////////

/**
   \brief creates a Camera
*/
Camera::Camera(void)
{
  this->setClassID(CL_CAMERA);
  this->target = new CameraTarget();

  this->setFovy(90);
  this->setAspectRatio(1.2f);
  this->setClipRegion(.1, 2000);

  this->setViewMode(VIEW_NORMAL);
}

/**
   \brief default destructor
*/
Camera::~Camera(void)
{
}

/**
   \brief focuses the Camera onto a Target
   \param target the new PNode the Camera should look at.
*/
void Camera::lookAt(PNode* target)
{
  this->target->setParent(target);
}

/**
   \returns The PNode of the Target (from there you can get position and so on
*/
PNode* Camera::getTarget(void)
{
  return (PNode*)this->target;
}


/**
   \brief sets a new AspectRatio
   \param aspectRatio the new aspect ratio to set (width / height)
*/
void Camera::setAspectRatio(float aspectRatio)
{
  this->aspectRatio = aspectRatio;
}

/**
   \brief sets the Field of View to fofy
   \param fovy new field of view factor (in degrees)
*/
void Camera::setFovy(float fovy)
{
  this->fovy = fovy;
}

/**
  \brief Sets a new clipping region
  \param nearClip The near clip plane
  \param farClip The far clip plane
*/
void Camera::setClipRegion(float nearClip, float farClip)
{
  this->nearClip = nearClip;
  this->farClip = farClip;
}

void Camera::setViewMode(ViewMode mode)
{
  switch (mode)
    {
    default:
    case VIEW_NORMAL:
      this->toFovy = 60.0;
      this->toRelCoor = Vector(-10, 5, 0);
      break;
    case VIEW_BEHIND:
      this->toFovy = 120.0;
      this->toRelCoor = Vector(-7, 0, 0);
      break;
    case VIEW_FRONT:
      this->toFovy = 95.0;
      this->toRelCoor = Vector(12, 5, 0);
      break;
    case VIEW_LEFT: 
      this->toFovy = 90;
      this->toRelCoor = Vector(0, 2, -10);
      break;
    case VIEW_RIGHT:
      this->toFovy = 90;
      this->toRelCoor = Vector(0, 2, 10);
      break;
    case VIEW_TOP:
      this->toFovy= 120;
      this->toRelCoor = Vector(0, 4, 0);
    }
}


/**
   \brief Updates the position of the camera.
   \param dt The time that elapsed.
*/
void Camera::tick(float dt)
{
  dt /= 500;
  float tmpFovy = (this->toFovy - this->fovy) * dt;
  if (tmpFovy > .001)
    this->fovy += (this->toFovy - this->fovy) * dt;
  Vector tmpPos = (this->toRelCoor - this->getRelCoor()) * dt;
  if (tmpPos.len() >= .001)
    {
      tmpPos = tmpPos + this->getRelCoor();
      this->setRelCoor(tmpPos);
    }
}


/**
   \brief initialize rendering perspective according to this camera
   
   This is called immediately before the rendering cycle starts, it sets all global
   rendering options as well as the GL_PROJECTION matrix according to the camera.
*/
void Camera::apply ()
{
  // switching to Projection Matrix
  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();

  // setting up the perspective
  gluPerspective(this->fovy,
		 this->aspectRatio,
		 this->nearClip,
		 this->farClip);

  // speed-up feature
  Vector cameraPosition = this->getAbsCoor();
  Vector targetPosition = this->target->getAbsCoor();
  Vector up = Vector(0, 1, 0);
  up = this->getAbsDir().apply(up);

  // Setting the Camera Eye, lookAt and up Vectors
  gluLookAt(cameraPosition.x, cameraPosition.y, cameraPosition.z,
	    targetPosition.x, targetPosition.y, targetPosition.z,
	    up.x, up.y, up.z);

  // switching back to Modeling Matrix
  glMatrixMode (GL_MODELVIEW);
}



///////////////////
// CAMERA-TARGET //
///////////////////


CameraTarget::CameraTarget()
{
  this->setClassID(CL_CAMERA_TARGET);
  this->setMode(PNODE_MOVEMENT);
}

CameraTarget::~CameraTarget()
{

}
