/*! 
    \file md2Model.h
    \brief Definition of an MD2 Model, a model format invented by ID Software.
    
    We are deeply thankfull for all the wunderfull things id software made for us gamers!
    
    The md2 file format is structured in a very simple way: it contains animations which are made out of
    frames (so called keyframes). Each frame is a complete draweable model in a specific position.
    A frame is a collection of vertex and its compagnions (normals, texture coordinates).
    
    A typical model has about 200 frames, the maximum frame count is fixed by MD2_MAX_FRAMES to currently 
    512 frames. The maximal vetices count is set to 2048 verteces, not enough?
    You just have to change the MD2_MAX_* values if it doesn't fit your purposes...
    
    Surface Culling is fully implemented in md2 models: quake2 uses front face culling.
*/

#ifndef _MD2MODEL_H
#define _MD2MODEL_H

#include "abstract_model.h"
#include "base_object.h"
#include "stdincl.h"


//! These are the needed defines for the max values when loading .MD2 files
#define MD2_IDENT                       (('2'<<24) + ('P'<<16) + ('D'<<8) + 'I')
#define MD2_VERSION                     8
#define MD2_MAX_TRIANGLES		4096
#define MD2_MAX_VERTICES		2048
#define MD2_MAX_TEXCOORDS		2048
#define MD2_MAX_FRAMES			512
#define MD2_MAX_SKINS			32
#define MD2_MAX_FRAMESIZE		(MD2_MAX_VERTICES * 4 + 128)

#define NUM_VERTEX_NORMALS              162
#define SHADEDOT_QUANT                  16

//! This stores the speed of the animation between each key frame - currently conflicting with the animation framework
#define kAnimationSpeed			12.0f

//! This holds the header information that is read in at the beginning of the file: id software definition
struct tMd2Header
{ 
   int ident;			        //!< This is used to identify the file
   int version;			        //!< The version number of the file (Must be 8)
    
   int skinWidth;			//!< The skin width in pixels
   int skinHeight;			//!< The skin height in pixels
   int frameSize;			//!< The size in bytes the frames are
   
   int numSkins;		        //!< The number of skins associated with the model
   int numVertices;			//!< The number of vertices (constant for each frame)
   int numTexCoords;			//!< The number of texture coordinates
   int numTriangles;			//!< The number of faces (polygons)
   int numGlCommands;			//!< The number of gl commands
   int numFrames;			//!< The number of animation frames
   
   int offsetSkins;			//!< The offset in the file for the skin data
   int offsetTexCoords;			//!< The offset in the file for the texture data
   int offsetTriangles;			//!< The offset in the file for the face data
   int offsetFrames;			//!< The offset in the file for the frames data
   int offsetGlCommands;		//!< The offset in the file for the gl commands data
   int offsetEnd;			//!< The end of the file offset
};


//! This is used to store the vertices that are read in for the current frame directly from the file (compressed)
struct tMd2AliasTriangle
{
   byte vertex[3];
   byte lightNormalIndex;
};

//! This stores the normals and vertices for the frames (uncompressed)
struct tMd2Triangle
{
   float vertex[3];
   float normal[3];
};

//! This stores the indices into the vertex and texture coordinate arrays
struct tMd2Face
{
   short vertexIndices[3];
   short textureIndices[3];
};

//! This stores UV coordinates
struct tMd2TexCoord
{
   short u, v;
};

//! This stores the animation scale, translation and name information for a frame, plus verts
struct tMd2AliasFrame
{
   float scale[3];
   float translate[3];
   char name[16];
   tMd2AliasTriangle aliasVertices[1];
};

//! This stores the frames vertices after they have been transformed
struct tMd2Frame
{
   char strName[16];
   tMd2Triangle *pVertices;
};

//! This stores a skin name
typedef char tMd2Skin[64];

typedef enum 
  {
    STAND,
    RUN,
    ATTACK,
    PAIN_A,
    PAIN_B,
    PAIN_C,
    JUMP,
    FLIP,
    SALUTE,
    FALLBACK,
    WAVE,
    POINT,
    CROUCH_STAND,
    CROUCH_WALK,
    CROUCH_ATTACK,
    CROUCH_PAIN,
    CROUCH_DEATH, 
    DEATH_FALLBACK,
    DEATH_FALLFORWARD,
    DEATH_FALLBACKSLOW,
    BOOM,
  
    MAX_ANIMATIONS
  } animType;



/* forward definitions */
class Material;




//! This is a MD2 Model class
class MD2Model : public AbstractModel {

public:
  MD2Model();
  virtual ~MD2Model();
  
  bool loadModel(const char* filename);
  bool loadSkin(const char* filename);
  
  void drawFrame(int frame);
  void draw();
  void draw2();
  
  void setAnim(int type);
  void scaleModel(float scaleFactor) { this->scaleFactor = scaleFactor;}

  void tick(float dtS);
  void debug();


private:
  void animate();  
  void processLighting();
  void interpolate(sVec3D* verticesList);
  void renderFrame();

 public:
  /* these variables are static, because they are all the same for every model */
  static sVec3D anorms[NUM_VERTEX_NORMALS];
  static float anormsDots[SHADEDOT_QUANT][256];
  static sAnim animationList[21];

 private:
  int numFrames;
  int numVertices;
  int numTriangles;
  int numGLCommands;
  char* fileName;
  char* skinFileName;
  tMd2Header* header;

  sVec3D* pVertices;
  int* pGLCommands;
  int* pLightNormals;
  glCommandVertex* pGLCommands2;

  unsigned int textureID;
  Material* material;
  sAnimState animationState;
  float scaleFactor;
};





#endif /* _MD2MODEL_H */
