
/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY


#include "skybox.h"

#include "stdincl.h"
#include "factory.h"

#include "material.h"
#include "vector.h"
#include "resource_manager.h"
#include "model.h"
//#include "world_entity.h"

CREATE_FACTORY(SkyBox);

using namespace std;

/**
   \brief Constructs a SkyBox and takes fileName as a map.
   \param fileName the file to take as input for the SkyBox
*/
SkyBox::SkyBox(const char* fileName)
{
  this->preInit();
  if (fileName)
    this->setTextureAndType(fileName, ".jpg");
  this->postInit();
}

SkyBox::SkyBox(const TiXmlElement* root) : WorldEntity(root)
{
  this->preInit();

  this->loadParams(root);

  this->postInit();
}

void SkyBox::loadParams(const TiXmlElement* root)
{
  LoadParam<SkyBox>(root, "Materialset", this, &SkyBox::setTexture);
}

void SkyBox::preInit(void)
{
  this->setClassName("SkyBox");
  this->skyModel = NULL;
  this->material = new Material*[6];
  for (int i = 0; i < 6; i++) 
    {
      this->material[i] = new Material();
      this->material[i]->setIllum(3);
      this->material[i]->setDiffuse(0.0,0.0,0.0);
      this->material[i]->setSpecular(0.0,0.0,0.0);
      this->material[i]->setAmbient(2.0, 2.0, 2.0);
    }
  this->setMode(PNODE_MOVEMENT);
}

void SkyBox::postInit(void)
{
  this->setSize(1900.0);
  this->rebuild();
}


/**
   \brief default destructor
*/
SkyBox::~SkyBox()
{
  PRINTF(5)("Deleting SkyBox\n");
  for (int i = 0; i < 6; i++)
    delete this->material[i];
  delete []this->material;
}

/**
   \brief sets A set of textures when just giving a Name and an extension:

   usage: give this function an argument like
   setTexture("skybox", "jpg");
   and it will convert this to 
   setTextures("skybox_top.jpg", "skybox_bottom.jpg", "skybox_left.jpg",
               "skybox_right.jpg", "skybox_front.jpg", "skybox_back.jpg");
*/
void SkyBox::setTextureAndType(const char* name, const char* extension)
{
  char* top    = new char[strlen(name)+strlen(extension)+ 10];
  char* bottom = new char[strlen(name)+strlen(extension)+ 10];
  char* left   = new char[strlen(name)+strlen(extension)+ 10];
  char* right  = new char[strlen(name)+strlen(extension)+ 10];
  char* front  = new char[strlen(name)+strlen(extension)+ 10];
  char* back   = new char[strlen(name)+strlen(extension)+ 10];

  sprintf(top, "%s_top.%s", name, extension);
  sprintf(bottom, "%s_bottom.%s", name, extension);
  sprintf(left, "%s_left.%s", name, extension);
  sprintf(right, "%s_right.%s", name, extension);
  sprintf(front, "%s_front.%s", name, extension);
  sprintf(back, "%s_back.%s", name, extension);
  
  this->setTextures(top, bottom, left, right, front, back);

  // deleted alocated memory of this function
  delete []top;
  delete []bottom;
  delete []left;
  delete []right;
  delete []front;
  delete []back;
}

/**
   \brief Defines which textures should be loaded onto the SkyBox.
   \param top the top texture.
   \param bottom the bottom texture.
   \param left the left texture.
   \param right the right texture.
   \param front the front texture.
   \param back the back texture.
*/
void SkyBox::setTextures(const char* top, const char* bottom, const char* left, const char* right, const char* front, const char* back)
{
  this->material[0]->setDiffuseMap(top);
  this->material[1]->setDiffuseMap(bottom);
  this->material[2]->setDiffuseMap(left);
  this->material[3]->setDiffuseMap(right);
  this->material[4]->setDiffuseMap(front);
  this->material[5]->setDiffuseMap(back);
}

/**
   \param size The new size of the SkyBox
*/
void SkyBox::setSize(float size)
{
  this->size = size;
}

/**
   \brief draws the SkyBox
*/
void SkyBox::draw()
{
  glPushMatrix();
  glMatrixMode(GL_MODELVIEW);
  Vector r = this->getAbsCoor();
  glTranslatef(r.x, r.y, r.z);

  this->skyModel->draw();

  glPopMatrix();
}


/**
   \brief rebuilds the SkyBox
   
   this must be done, when changing the Size of the Skybox (runtime-efficency)
*/
void SkyBox::rebuild()
{
  if (this->skyModel)
    delete skyModel;
  skyModel = new Model();

  this->skyModel->addVertex (-0.5*size, -0.5*size, 0.5*size);  
  this->skyModel->addVertex (0.5*size, -0.5*size, 0.5*size);
  this->skyModel->addVertex (-0.5*size, 0.5*size, 0.5*size);
  this->skyModel->addVertex (0.5*size, 0.5*size, 0.5*size);
  this->skyModel->addVertex (-0.5*size, 0.5*size, -0.5*size);
  this->skyModel->addVertex (0.5*size, 0.5*size, -0.5*size);
  this->skyModel->addVertex (-0.5*size, -0.5*size, -0.5*size);
  this->skyModel->addVertex (0.5*size, -0.5*size, -0.5*size);

  this->skyModel->addVertexTexture (0.0, 0.0);
  this->skyModel->addVertexTexture (1.0, 0.0);
  this->skyModel->addVertexTexture (1.0, 1.0);
  this->skyModel->addVertexTexture (0.0, 1.0);

  this->skyModel->addVertexNormal (0.0, 0.0, 1.0);
  this->skyModel->addVertexNormal (0.0, 1.0, 0.0);
  this->skyModel->addVertexNormal (0.0, 0.0, -1.0);
  this->skyModel->addVertexNormal (0.0, -1.0, 0.0);
  this->skyModel->addVertexNormal (1.0, 0.0, 0.0);
  this->skyModel->addVertexNormal (-1.0, 0.0, 0.0);

  this->skyModel->setMaterial(material[0]);
  this->skyModel->addFace (4, VERTEX_TEXCOORD_NORMAL, 2,0,3, 3,1,3, 5,2,3, 4,3,3); // top
  this->skyModel->setMaterial(material[1]);
  this->skyModel->addFace (4, VERTEX_TEXCOORD_NORMAL, 6,0,1, 7,1,1, 1,2,1, 0,3,1); // bottom
  this->skyModel->setMaterial(material[2]);
  this->skyModel->addFace (4, VERTEX_TEXCOORD_NORMAL, 4,2,2, 5,3,2, 7,0,2, 6,1,2); // left
  this->skyModel->setMaterial(material[3]);
  this->skyModel->addFace (4, VERTEX_TEXCOORD_NORMAL, 0,0,0, 1,1,0, 3,2,0, 2,3,0); // right
  this->skyModel->setMaterial(material[4]);
  this->skyModel->addFace (4, VERTEX_TEXCOORD_NORMAL, 1,0,5, 7,1,5, 5,2,5, 3,3,5); // front
  this->skyModel->setMaterial(material[5]);
  this->skyModel->addFace (4, VERTEX_TEXCOORD_NORMAL, 6,0,4, 0,1,4, 2,2,4, 4,3,4); // back
  
  this->skyModel->finalize();
}
