/*! 
    \file world.h
    \brief Holds and manages all game data
*/ 

#ifndef _WORLD_H
#define _WORLD_H

#include "stdincl.h"
#include "comincl.h"
#include "story_entity.h"
#include "p_node.h"

class World;
class WorldEntity;
class TrackManager;
class Camera;
class PNode;
class GLMenuImageScreen;
class LightManager;
class ParticleEngine;
class Terrain;
class GarbageCollector;
class Text;
class TiXmlElement;

//! The game world Interface
/**
   this is a singleton interface, that enables world_entities to access the
   world. for those objects, there is no easier way than over this interface!
*/
class WorldInterface : BaseObject {

 public:
  ~WorldInterface();
  static WorldInterface* getInstance();
  void init(World* world);
  tList<WorldEntity>* getEntityList();

 private:
  WorldInterface();
  static WorldInterface* singletonRef;    //!< singleton reference to this object
  bool worldIsInitialized;                //!< true if the world has been initialized
  World* worldReference;                  //!< this is a reference to the running world

};

//! The game world
/**
   this class initializes everything that should be displayed inside of the current level.
   it is the main driving factor during gameplay.
*/
class World : public StoryEntity {

 public:
  World (char* name);
  World (int worldID);
  World (const TiXmlElement* root = NULL);
  virtual ~World ();

  void loadParams(const TiXmlElement* root);

  double getGameTime();

  /* classes from story-entity */
  virtual ErrorMessage preLoad();
  virtual ErrorMessage load ();
  virtual ErrorMessage init ();
  virtual ErrorMessage start ();
  virtual ErrorMessage stop ();
  virtual ErrorMessage pause ();
  virtual ErrorMessage resume ();
  virtual ErrorMessage destroy ();

  void loadDebugWorld(int worldID);

  virtual void displayLoadScreen();
  virtual void releaseLoadScreen();
  
  /* command node functions */
  bool command (Command* cmd);

  tList<WorldEntity>* getEntities();

  /* interface to world */
  void spawn (WorldEntity* entity);
  void spawn (WorldEntity* entity, Vector* absCoor, Quaternion* absDir);
  void spawn(WorldEntity* entity, PNode* parentNode, Vector* relCoor, Quaternion* relDir, 
	     int parentingMode);

  const char* getPath();
  void setPath( const char* name);

 private:
  void constuctorInit(char* name, int worldID);

  Uint32 lastFrame;                   //!< last time of frame
  Uint32 dt;                          //!< time needed to calculate this frame
  float dtS;                          //!< The time needed for caluculations in seconds
  double gameTime;                    //!< this is where the game time is saved
  bool bQuitOrxonox;                  //!< quit this application
  bool bQuitCurrentGame;              //!< quit only the current game and return to menu
  bool bPause;                        //!< pause mode

  GLMenuImageScreen* glmis;           //!< The Level-Loader Display

  char* worldName;                    //!< The name of this World
  int debugWorldNr;                   //!< The Debug Nr. needed, if something goes wrong
  char* path;                         //!< The file from which this world is loaded

  PNode* nullParent;                  //!< The zero-point, that everything has as its parent.
  TrackManager* trackManager;         //!< The reference of the TrackManager that handles the course through the Level.
  ParticleEngine* particleEngine;     //!< The ParticleEngine of the World.
  Camera* localCamera;                //!< The current Camera
  WorldEntity* sky;                   //!< The Environmental Heaven of orxonox \todo insert this to environment insted
  LightManager* lightMan;             //!< The Lights of the Level
  Terrain* terrain;                   //!< The Terrain of the World.

  GLuint objectList;                  //!< temporary: \todo this will be ereased soon
  tList<WorldEntity>* entities;       //!< A template List of all entities. Every moving thing should be included here, and world automatically updates them.
  WorldEntity* localPlayer;           //!< The Player, you fly through the level.

  GarbageCollector* garbageCollector; //!< reference to the garbage  collector

  /* function for main-loop */
  void mainLoop ();
  void synchronize ();
  void handleInput ();
  void tick ();
  void update ();
  void collide ();
  void draw ();
  void display ();
  void debug ();

};

#endif /* _WORLD_H */
