/*! 
    \file particle_system.h

*/

#ifndef _PARTICLE_SYSTEM_H
#define _PARTICLE_SYSTEM_H

#include "base_object.h"
#include "vector.h"

#define PARTICLE_DOT_MASK           0x000001
#define PARTICLE_SPARK_MASK         0x000010
#define PARTICLE_SPRITE_MASK        0x000100
#define PARTICLE_MODEL_MASK         0x001000
#define PARTICLE_WORDL_ENTITY_MASK  0x010000
#define PARTICLE_MULTI_MASK         0x100000

//! An enumerator for the different types of particles.
typedef enum PARTICLE_TYPE {PARTICLE_DOT = PARTICLE_DOT_MASK,
			    PARTICLE_SPARK = PARTICLE_SPARK_MASK,
			    PARTICLE_SPRITE = PARTICLE_SPRITE_MASK,
			    PARTICLE_MULTI_SPRITE = PARTICLE_SPRITE_MASK | PARTICLE_MULTI_MASK,
			    PARTICLE_MODEL = PARTICLE_MODEL_MASK,
			    PARTICLE_MULTI_MODE = PARTICLE_MODEL_MASK | PARTICLE_MULTI_MASK};

#define PARTICLE_DEFAULT_MAX_COUNT    200               //!< a default count of particles in the system.
#define PARTICLE_DEFAULT_TYPE         PARTICLE_SPRITE   //!< A default type of the system.

// FORWARD DEFINITION
class Material;
class ParticleEmitter;


//! A struct for one Particle
typedef struct Particle
{
  float timeToLive;           //!< The time this particle lives from NOW on.
  Vector position;            //!< The current position of this particle.
  Vector velocity;            //!< The current velocity of this particle.
  Quaternion rotation;        //!< The current rotation of this particle.
  float mass;                 //!< The mass of this particle.
  float radius;               //!< The current size of this particle.
  float radiusIt;             //!< The difference of the Size per second.

  PARTICLE_TYPE type;

  Particle* next;             //!< pointer to the next particle in the List. (NULL if no preceding one)
};

//! A class to handle particle Systems
class ParticleSystem : public BaseObject {
  friend class ParticleEmitter;

 public:
  ParticleSystem(unsigned int maxCount = PARTICLE_DEFAULT_MAX_COUNT, PARTICLE_TYPE type = PARTICLE_DEFAULT_TYPE);
  virtual ~ParticleSystem();
  void setName(const char* name);
  const char* getName(void) const;

  void setType(PARTICLE_TYPE particleType, int count = 0);
  void setMaterial(Material* material);
  void setInheritSpeed(float value);
  void setLifeSpan(float lifeSpan, float randomLifeSpan = 0.0);
  void setRadius(float startRadius, float endRadius, float randomStartRadius = 0.0, float randomEndRadius = 0.0);
  void setConserve(float conserve);
  void setMass(float mass, float randomMass);

  void tick(float dt);
  void draw(void);

  void debug(void);

 private:
  char* name;                // the Name of the Particle System

  float conserve;            //!< How much energy gets conserved to the next Tick.
  float lifeSpan;            //!< Initial lifetime of a Particle.
  float randomLifeSpan;
  float startRadius;
  float endRadius;
  float randomStartRadius;
  float randomEndRadius;
  float initialMass;
  float randomInitialMass;
  float inheritSpeed;

  // particles
  int maxCount;              //!< The maximum count of Particles.
  int count;                 //!< The current count of Particles.
  PARTICLE_TYPE particleType;//!< A type for all the Particles
  Material* material;        //!< A Material for all the Particles.
  Particle* particles;       //!< A list of particles of this System.
  Particle* deadList;        //!< A list of dead Particles in the System.

  GLuint* glID;              //!< A List of different gl-List-ID's
  GLuint dialectCount;       //!< How many different types of particles are there in the Particle System

  void addParticle(const Vector& position, const Vector& velocity, unsigned int data = 0);
  
};

#endif /* _PARTICLE_SYSTEM_H */
