/*!
  \file model.h
  \brief Contains the Model Class that handles 3D-Models
*/

#ifndef _MODEL_H
#define _MODEL_H

#include "material.h"
#include "glincl.h"

// FORWARD DEFINITION //
class Array;
class Vector;
template<class T> class tList;

//! an enumerator fot the different Model Types.
/**
   MODEL_DISPLAY_LIST means, that a DisplayList will be built out of the model. This model will be STATIC, meaning it cannot be changed after initialisation.
   MODEL_VERTEX_ARRAY means, that a VertexArray will be built out of the model. This moel will be DYNAMIX, meaning that one can change the properties from outside of the model.
*/
typedef enum MODEL_TYPE {MODEL_DISPLAY_LIST,
			 MODEL_VERTEX_ARRAY};


// definition of different modes for setting up Faces
#define VERTEX 0       //!< If Faces are created WITH Vertex-Coordinate
#define NORMAL 1       //!< If Faces are created WITH Normals (otherwise autocalculate)
#define TEXCOORD 2     //!< If Faces are created WITH TextureCoordinate
//! an enumerator for VERTEX_FORMAT
typedef enum VERTEX_FORMAT {VERTEX_ONLY = VERTEX,
		    VERTEX_NORMAL = NORMAL,
		    VERTEX_TEXCOORD = TEXCOORD,
		    VERTEX_TEXCOORD_NORMAL = NORMAL | TEXCOORD};

////////////////////
/// SUB-ELEMENTS ///
////////////////////
//! This is the placeholder of one Vertex beloning to a Face.
class ModelFaceElement
{
 public:
  ModelFaceElement();
  ~ModelFaceElement();

  int vertexNumber;         //!< The number of the Vertex out of the Array* vertices, this vertex points to.
  int normalNumber;         //!< The number of the Normal out of the Array* normals, this vertex points to.
  int texCoordNumber;       //!< The number of the textureCoordinate out of the Array* vTexture, this vertex points to.
  ModelFaceElement* next;   //!< Point to the next FaceElement in this List.
};

//! This is the placeholder of a Face belonging to a Group of Faces.
class ModelFace
{
 public:
  ModelFace();
  ~ModelFace();

  int vertexCount;                //!< The Count of vertices this Face has.
  ModelFaceElement* firstElem;    //!< Points to the first Vertex (FaceElement) of this Face.
  
  Material* material;             //!< The Material to use.
  
  ModelFace* next;                //!< Pointer to the next Face.
}; 

//! Group to handle multiple Models per obj-file.
class ModelGroup
{
 public:
  ModelGroup();
  ~ModelGroup();

  void cleanup();

  char* name;                 //!< the Name of the Group. this is an identifier, that can be accessed via the draw (char* name) function.
  
  GLubyte* indices;           //!< The indices of the Groups. Needed for vertex-arrays
  GLuint listNumber;          //!< The number of the GL-List this Group gets.
  ModelFace* firstFace;       //!< The first Face in this group.
  ModelFace* currentFace;     //!< The current Face in this Group (the one we are currently working with.)
  int faceMode;               //!< The Mode the Face is in: initially -1, 0 for FaceList opened, 1 for Material,  3 for triangle, 4 for Quad, 5+ for Poly \todo ENUM...
  int faceCount;              //!< The Number of Faces this Group holds.
  
  ModelGroup* next;           //!< Pointer to the next Group.
};

/////////////
/// MODEL ///
/////////////

//! Class that handles 3D-Models. it can also read them in and display them.
class Model
{
 private:

  char* name;                 //!< This is the name of the Model.
  MODEL_TYPE type;
  bool finalized;             //!< Sets the Object to be finalized.

  Array* vertices;            //!< The Array that handles the Vertices.
  int verticesCount;          //!< A global Counter for vertices.
  Array* normals;             //!< The Array that handles the Normals.
  Array* vTexture;            //!< The Array that handles the VertexTextureCoordinates.

  ModelGroup* firstGroup;     //!< The first of all groups.
  ModelGroup* currentGroup;   //!< The currentGroup. this is the one we will work with.
  int groupCount;             //!< The Count of Groups.

  tList<Material>* materialList;//!< A list for all the Materials in this Model
  
  bool buildVertexNormals(void);

  bool importToDisplayList(void);
  bool addGLElement(ModelFaceElement* elem);

  bool importToVertexArray(void);

  bool deleteArrays(void);
  bool cleanup(void);


 protected:
  float scaleFactor;    //!< The Factor with which the Model should be scaled. \todo maybe one wants to scale the Model after Initialisation

  Material* findMaterialByName(const char* materialName);

  void cubeModel(void);

 public:
  Model(const char* modelName = NULL, MODEL_TYPE type = MODEL_DISPLAY_LIST);
  virtual ~Model(void);

  void setName(const char* name);
  inline const char* getName() {return this->name;}
  
  void draw(void) const;
  void draw(int groupNumber) const;
  void draw(char* groupName) const;
  int getGroupCount() const;

  Material* addMaterial(Material* material);
  Material* addMaterial(const char* materialName);

  bool addGroup(const char* groupString);
  bool addVertex(const char* vertexString);
  bool addVertex(float x, float y, float z);
  bool addFace(const char* faceString);
  bool addFace(int faceElemCount, VERTEX_FORMAT type, ...);
  bool addVertexNormal(const char* normalString);
  bool addVertexNormal(float x, float y, float z);
  bool addVertexTexture(const char* vTextureString);
  bool addVertexTexture(float u, float v);
  bool setMaterial(const char* mtlString);
  bool setMaterial(Material* mtl);
  void finalize(void);
};

#endif
