/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "objModel.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define PARSELINELENGTH 8192

#include <fstream>

#include "debug.h"

/**
   \brief Crates a 3D-Model, loads in a File and scales it.
   \param fileName file to parse and load (must be a .obj file)
   \param scaling The factor that the model will be scaled with.
*/
OBJModel::OBJModel(char* fileName, float scaling)
{
  this->initializeOBJ();
  this->scaleFactor = scaling;

  this->importFile (fileName);

  this->importToGL ();

  this->cleanup();
}

/**
   \brief deletes an OBJModel.

   Looks if any from model allocated space is still in use, and if so deleted it.
*/
OBJModel::~OBJModel()
{
  PRINTF(4)("Deleting the obj-names\n");
  if (this->objPath)
    delete []this->objPath;
  if (this->objFileName)
    delete []this->objFileName;
  if (this->mtlFileName)
    delete []this->mtlFileName;
}

/**
   \brief Initializes an obj-model
*/
void OBJModel::initializeOBJ(void)
{
  this->objPath = NULL;
  this->objFileName = NULL;
  this->mtlFileName = NULL;

  this->initialize();
}

/**
   \brief Imports a obj file and handles the the relative location
   \param fileName The file to import
*/
bool OBJModel::importFile (char* fileName)
{
  PRINTF(4)("preparing to read in file: %s\n", fileName);


#ifdef __WIN32__
  // win32 path reading
  char pathSplitter= '\\';
#else /* __WIN32__ */
  // unix path reading
  char pathSplitter='/';
#endif /* __WIN32__ */
  char* tmpName = fileName;
  if (tmpName[0] == pathSplitter)
    tmpName++;
  char* name = tmpName;
  while (( tmpName = strchr (tmpName+1, pathSplitter)))
    {
      name = tmpName+1;
    }
  this->objPath = new char[name-fileName+1];
  strncpy(this->objPath, fileName, name-fileName);
  this->objPath[name-fileName] = '\0';
  if (strlen(objPath)> 0)
    PRINTF(5)("Resolved file %s to folder: %s.\n", name, objPath);
  else
    PRINTF(5)("Resolved file %s.\n", name);
  
  this->setName(name);
  if (this->material)
    this->material->addTexturePath(this->objPath);
  this->objFileName = new char[strlen(name)+1];
  strcpy (this->objFileName, name);
  this->readFromObjFile ();
  return true;
}

/**
   \brief Reads in the .obj File and sets all the Values.
   This function does read the file, parses it for the occurence of things like vertices, faces and so on, and executes the specific tasks
*/
bool OBJModel::readFromObjFile (void)
{
  char* fileName = new char [strlen(objPath)+strlen(objFileName)+1];
  if (this->objFileName != NULL && !strcmp(this->objFileName, ""))
    return false;
  strcpy(fileName, this->objPath);
  strcat(fileName, this->objFileName);

  FILE* stream;
  if( (stream = fopen (fileName, "r")) == NULL)
    {
      printf("IniParser could not open %s\n", fileName);
      return false;
    }

  char readLine[PARSELINELENGTH];
  char buffer[PARSELINELENGTH];
  int i = 0;
  while(fgets(readLine, PARSELINELENGTH, stream))
    { 
      strcpy(buffer, readLine);
      i++;
      // case vertice
      if (!strncmp(buffer, "v ", 2))
	{
	  this->addVertex(buffer+2);
	}

      // case face
      else if (!strncmp(buffer, "f ", 2))
	{
	  this->addFace (buffer+2);
	}
      
      else if (!strncmp(buffer, "mtllib ", 7))
	{
	  this->readMtlLib (buffer+7);
	}

      else if (!strncmp(buffer, "usemtl ", 7))
	{
	  this->addUseMtl (buffer+7);
	}

      // case VertexNormal
      else if (!strncmp(buffer, "vn ", 3))
	{
	  this->addVertexNormal(buffer+3);
	}
      
      // case VertexTextureCoordinate
      else if (!strncmp(buffer, "vt ", 3))
	{
	  this->addVertexTexture(buffer+3);
	}
      // case group
      else if (!strncmp(buffer, "g ", 2))
	{
	  this->addGroup (buffer+2);
	}
      else if (!strncmp(buffer, "s ", 2)) //! \todo smoothing groups have to be implemented
	{
	  PRINTF(2)("smoothing groups not supportet yet. line: %s\n", buffer);
	}
    }
  fclose (stream);
  delete []fileName;
  return true;
}

/** 
    \brief Function to read in a mtl File.
    \param mtlFile The .mtl file to read

    This Function parses all Lines of an mtl File.
    The reason for it not to be in the materials-class is,
    that a material does not have to be able to read itself in from a File.

*/
bool OBJModel::readMtlLib (char* mtlFile)
{
  this->mtlFileName = new char [strlen(mtlFile)+1];
  strcpy(this->mtlFileName, mtlFile);
  char* fileName = new char [strlen(objPath) + strlen(this->mtlFileName)+1];
  strcpy(fileName, this->objPath);
  strcat(fileName, this->mtlFileName);
  

  PRINTF(4)("Opening mtlFile: %s\n", fileName);

  ifstream* MTL_FILE = new ifstream (fileName);
  if (MTL_FILE->fail())
    {
      PRINTF(2)("unable to open file: %s\n", fileName);
      MTL_FILE->close();
      delete []fileName;
      delete MTL_FILE;
      return false;
    }
  char buffer[500];
  Material* tmpMat = material;
  while(!MTL_FILE->eof())
    {
      MTL_FILE->getline(buffer, 500);
      PRINTF(5)("found line in mtlFile: %s\n", buffer);
      

      // create new Material
      if (!strncmp(buffer, "newmtl ", 7))
	{
	  tmpMat = tmpMat->addMaterial(buffer+7);
	  //	  PRINTF(2)("%s, %p\n", tmpMat->getName(), tmpMat);
	}
      // setting a illumMode
      else if (!strncmp(buffer, "illum ", 6))
	{
	  tmpMat->setIllum(buffer+6);

	}
      // setting Diffuse Color
      else if (!strncmp(buffer, "Kd ", 3))
	{
	  tmpMat->setDiffuse(buffer+3);
	}
      // setting Ambient Color
      else if (!strncmp(buffer, "Ka ", 3))
	{
	  tmpMat->setAmbient(buffer+3);
	}
      // setting Specular Color
      else if (!strncmp(buffer, "Ks ", 3))
	{
	  tmpMat->setSpecular(buffer+3);
	}
      // setting The Specular Shininess
      else if (!strncmp(buffer, "Ns ", 3))
	{
	  tmpMat->setShininess(buffer+3);
	}
      // setting up transparency
      else if (!strncmp(buffer, "d ", 2))
	{
	  tmpMat->setTransparency(buffer+2);
	}
      else if (!strncmp(buffer, "Tf ", 3))
	{
	  tmpMat->setTransparency(buffer+3);
	}
      
      else if (!strncmp(buffer, "map_Kd ", 7))
	{
	  tmpMat->setDiffuseMap(buffer+7);
	}
      else if (!strncmp(buffer, "map_Ka ", 7))
	{
	  tmpMat->setAmbientMap(buffer+7);
	}
      else if (!strncmp(buffer, "map_Ks ", 7))
	{
	  tmpMat->setSpecularMap(buffer+7);
	}
      else if (!strncmp(buffer, "bump ", 5))
	{
	  tmpMat->setBump(buffer+7);
	}
     

    }
  MTL_FILE->close();
  delete []fileName;
  delete MTL_FILE;
  return true;
}
