/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "texture.h"

#include "graphics_engine.h"

/**
   \brief Constructor for a Texture
*/
Texture::Texture(void)
{
  this->pImage = new Image;
  this->pImage->data = NULL;
  this->map = NULL;
  this->texture = 0;
}

/**
   \brief Constructor for a Texture
*/
Texture::Texture(const char* imageName)
{
  this->pImage = new Image;
  this->pImage->data = NULL;
  this->map = NULL;
  this->texture = 0;
  this->loadImage(imageName);
}  

/**
   \brief Destructor of a Texture
   
   Frees Data, and deletes the textures from GL
*/
Texture::~Texture(void)
{
  if (this->pImage->data)
    delete []this->pImage->data;
  delete pImage;
  if (this->texture)
    glDeleteTextures(1, &this->texture);
}

/**
   \brief a Simple function that switches two char values
   \param a The first value
   \param b The second value
*/
void Texture::swap (unsigned char &a, unsigned char &b)
{
  unsigned char temp;
  temp = a;
  a    = b;
  b    = temp;
}


/**
   \brief Loads a Texture to the openGL-environment.
   \param pImage The Image to load to openGL
*/
bool Texture::loadTexToGL (Image* pImage)
{
  if (GraphicsEngine::texturesEnabled)
    {
      PRINTF(4)("Loading texture to OpenGL-Environment.\n");
      glGenTextures(1, &this->texture);
      glBindTexture(GL_TEXTURE_2D, this->texture);
      /* not Working, and not needed. 
	 glTexImage2D( GL_TEXTURE_2D, 0, 3, width,
	 height, 0, GL_BGR,
	 GL_UNSIGNED_BYTE, map->pixels );
      */ 
      gluBuild2DMipmaps(GL_TEXTURE_2D, 3, pImage->width, pImage->height, pImage->format, GL_UNSIGNED_BYTE, pImage->data);
      
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_NEAREST);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR_MIPMAP_LINEAR);	
    }
}

/**
   \brief loads an Image from a file to a Texture
   \param imageName The image to load
*/
bool Texture::loadImage(const char* imageName)
{
  if (GraphicsEngine::texturesEnabled)
    {
      if (imageName)
	{
	  this->map=IMG_Load(imageName);
	  if(!map)
	    {
	      PRINTF(1)("IMG_Load: %s\n", IMG_GetError());
	      return false;
	    }
	  pImage->height = map->h;
	  pImage->width  = map->w;
	  pImage->data   = (GLubyte*)map->pixels;
	  pImage->bpp    = map->format->BytesPerPixel;
	  if (pImage->bpp == 3)
	    pImage->format = GL_RGB;
	  else if (pImage->bpp == 4)
	    {
	      pImage->format = GL_RGBA;
	      SDL_SetAlpha(this->map, 0, 0);
	    }

	  if( !IMG_isPNG(SDL_RWFromFile(imageName, "rb")) && !IMG_isJPG(SDL_RWFromFile(imageName, "rb")))
	    for (int i=0;i<map->h * map->w *3;i+=3)
	      { 
		GLuint temp = pImage->data[i];
		pImage->data[i] = pImage->data[i+2];
		pImage->data[i+2] = temp;
	      }
	  /* this is the real swapping algorithm */
	  for( int i = 0 ; i < (pImage->height / 2) ; ++i )
	    for( int j = 0 ; j < pImage->width * pImage->bpp; j += pImage->bpp )
	      for(int k = 0; k < pImage->bpp; ++k)
		swap( pImage->data[ (i * pImage->width * pImage->bpp) + j + k], pImage->data[ ( (pImage->height - i - 1) * pImage->width * pImage->bpp ) + j + k]);
	  
	  this->loadTexToGL (this->pImage);
	  SDL_FreeSurface(map);
	  pImage->data = NULL;
	}
      else
	{
	  PRINTF(2)("Image not Found: %s\n", imageName);
	  return false;
	}
    }
}
