/*! 
    \file text_engine.h
    \brief Definition of textEngine, the Font and the Text

    Text is the text outputed.
    Font is a class that loads a certain ttf-file with a specific height into memory
    TextEngine is used to manage the all the different Fonts that might be included

    for more information see the specific classes.

    !! IMPORTANT !! When using ttf fonts clear the license issues prior to
   adding them to orxonox. This is really important, because we do not want
   to offend anyone.
*/

#ifndef _TEXT_ENGINE_H
#define _TEXT_ENGINE_H


#include "glincl.h"
#include "SDL_ttf.h"

#include "vector.h"
#include "base_object.h"

// FORWARD DECLARATION
class PNode;
class Font;
template<class T> class tList;

//! An enumerator for the text alignment.
enum TEXT_ALIGNMENT {TEXT_ALIGN_LEFT, TEXT_ALIGN_RIGHT, TEXT_ALIGN_CENTER, TEXT_ALIGN_SCREEN_CENTER};

/* some default values */
#define FONT_DEFAULT_SIZE       50                   //!< default size of the Text
#define FONT_DEFAULT_TEXT	"orxonox 1234567890" //!< default text to display
#define FONT_DEFAULT_COLOR_R    255                  //!< default red part (color) of the text
#define FONT_DEFAULT_COLOR_G    255                  //!< default red green (color) of the text
#define FONT_DEFAULT_COLOR_B    255                  //!< default red blue (color) of the text
#define FONT_NUM_COLORS         256                  //!< number of colors.

#define FONT_HIGHEST_KNOWN_CHAR 128                  //!< The highest character known to the textEngine.

#define TEXT_DEFAULT_ALIGNMENT  TEXT_ALIGN_CENTER    //!< default alignment
#define TEXT_STATIC             0                    //!< Static Text
#define TEXT_DYNAMIC            1                    //!< Dynamic Text
/**
 * STATIC means: a font, that is only one GL-face.
 ** it is very fast, and can be used for all text
 ** that does not have to be changed anymore, or if
 ** the the text should look very nice
 * DYNAMIC means: a very fast font, that will is build
 ** from multiple quads.
 ** Use this type, if you want to create fast changing
 ** text like a counter.
 */


//! A Struct to handel Texture Coordinates for quads
struct TexCoord
{
  float minU;                      //!< The minimum U-Coordinate
  float maxU;                      //!< The maximum U-Coordinate
  float minV;                      //!< The minimum V-Coordinate
  float maxV;                      //!< The maximum V-Coordinate
};

//! A struct for handling glyphs
/**
   a Glyph is one letter of a certain font
*/
struct Glyph
{
  // Glyph-specific (size and so on)
  Uint16 character;              //!< The character
  int minX;                      //!< The minimum distance from the origin in X
  int maxX;                      //!< The maximum distance from the origin in X
  int minY;                      //!< The minimum distance from the origin in Y
  int maxY;                      //!< The maximum distance from the origin in Y
  int width;                     //!< The width of the Glyph
  int height;                    //!< The height of the Glyph
  int bearingX;                  //!< How much is right of the Origin
  int bearingY;                  //!< How much is above the Origin
  int advance;                   //!< How big a Glyph would be in monospace-mode
  
  // OpenGL-specific
  //  TexCoord texCoord;         //!< A Texture Coordinate for this glyph.
  GLuint displayList;            //!< DiplayList to render this Glyph.
};

////////////
/// TEXT ///
////////////
//! Represents one textElement.
class Text : public BaseObject
{
  friend class TextEngine;
 public:
  ~Text(void);

  void setBindNode(PNode* bindNode);

  void setType(int type);
  void setText(const char* text);
  void setPosition(int x, int y);
  void setAlignment(TEXT_ALIGNMENT alignment);
  /** \param blending the blending intensity to set (between 0.0 and 1.0) */
  inline void setBlending(float blending) {this->blending = blending;}

  // Static Text
  void setColor(Uint8 r, Uint8 g, Uint8 b);
  void setStyle(char* renderStyle);

  void createTexture();

  void draw(void) const;
  
  void debug(void) const;

 private:
  Text(Font* font, int type = TEXT_DYNAMIC);

  Font* font;

  int type;                      //!< The type of this Font.
  char* text;                    //!< The text to display
  SDL_Color color;               //!< The color of the font.
  TEXT_ALIGNMENT alignment;      //!< The aignment of the text.
  float blending;                //!< The blending intensity.
  // placement in openGL
  GLuint texture;                //!< A GL-texture to hold the text
  TexCoord texCoord;             //!< Texture-coordinates \todo fix this to have a struct
  SDL_Rect posSize;              //!< An SDL-Rectangle representing the position and size of the Text on the screen.
  
  PNode* bindNode;               //!< A node the Text is bind to. (if NULL thr node will not be bound to anything.)
};

////////////
/// FONT ///
////////////
//! A class to handle a Font of a certain ttf-File, Size and Color.
class Font
{
  friend class Text;
 public:
  Font(const char* fontFile, unsigned int fontSize = FONT_DEFAULT_SIZE,
       Uint8 r = FONT_DEFAULT_COLOR_R, Uint8 g = FONT_DEFAULT_COLOR_G, Uint8 b = FONT_DEFAULT_COLOR_B);
  virtual ~Font();

  // font
  bool setFont(const char* fontFile);
  void setSize(unsigned int fontSize);
  void setFastColor(Uint8 r, Uint8 g, Uint8 b);
  void setStyle(char* renderStyle);

  inline Glyph** getGlyphArray(void) {return glyphArray;}
  inline GLuint getFastTextureID(void) {return fastTextureID;}
  
  
 private:
  // general purpose
  GLdouble projMat[16];             //!< The Projection Matrix

  // information about the Font
  TTF_Font* font;                   //!< The font we use for this.
  char* fontFile;                   //!< The fontfile from whitch the font was loaded.
  unsigned int fontSize;            //!< The size of the font in pixels. each Font has one size.
  int renderStyle;                  //!< The Renderstyle
  
  Glyph** glyphArray;               //!< An Array of all the Glyphs stored in the Array of Glyphs.
  GLuint fastTextureID;             //!< The fast textureID.
  SDL_Color fastColor;              //!< A Color for the fast Texture.

  tList<Text>* textList;

  int getMaxHeight(void);
  int getMaxAscent(void);
  int getMaxDescent(void);
  Glyph* getGlyphMetrics(Uint16 character);

  GLuint createFastTexture();

  void initGlyphs(Uint16 from, Uint16 count);
  int findOptimalFastTextureSize(void);

  void debug(void);

};
GLuint loadTexture(SDL_Surface* surface, TexCoord* texCoord);
int powerOfTwo(int input);


///////////////////
/// TEXT-ENGINE ///
///////////////////
//! A singleton Class that operates as a Handler for generating and rendering Text in 2D
class TextEngine : public BaseObject 
{
 public:
  static TextEngine* getInstance(void);
  virtual ~TextEngine(void);

  Text* createText(const char* fontFile,
		   unsigned int fontSize = FONT_DEFAULT_SIZE,
		   int textType = TEXT_DYNAMIC,
		   Uint8 r = FONT_DEFAULT_COLOR_R,
		   Uint8 g = FONT_DEFAULT_COLOR_G,
		   Uint8 b = FONT_DEFAULT_COLOR_B);
  
  void deleteText(Text* text);
  void flush(void);

  void draw(void) const;
 
  void debug(void) const;

 private:
  TextEngine(void);
  static TextEngine* singletonRef;

  // general
  static void enableFonts(void);
  static void disableFonts(void);
  static bool checkVersion(void);


  //  tList<Font>* fontList;
  tList<Text>* textList;

};

#endif /* _TEXT_ENGINE_H */
