/*! 
    \file track_manager.h
    \brief manages all tracks defined in the world and the path the player takes

    it is a container for all tracks and all track-nodes. it manages the movement of
    the track helper-parent (that drives the player). it is responsable for calculating
    smooth curves etc.
*/


#ifndef _TRACK_MANAGER_H
#define _TRACK_MANAGER_H

#include "curve.h"
#include "base_object.h"

class PNode;
template<class T> class tList;

// Static Definitions

//! The Default Curve-Type to set for the whole path (if not chosen otherwise).
#define TMAN_DEFAULT_CURVETYPE CURVE_BEZIER
#define TMAN_DEFAULT_DURATION 10
#define TMAN_DEFAULT_WIDTH    10

//! A Graph-Element, that holds the curve-structure of a Level.
/**
   A TrackElement is used, to define the structure of the Track itself.
   It is a graph and not a tree, because paths can fork and join again.
*/
class TrackElement
{
 public:
  TrackElement(void);
  ~TrackElement(void);

  TrackElement* findByID(unsigned int trackID);
  TrackElement* findByName(const char* trackName);
  bool backLoopCheck(const TrackElement* trackElem, unsigned int depth = 0) const;

  TrackElement* getChild(int childNumber) const;
  void setName(const char* name);
  const char* getName(void) const;

  // atributes
  bool isFresh;              //!< If no Points where added until now
  bool isHotPoint;           //!< If the first node is a specialPoint;
  bool isSavePoint;          //!< If the first node is a savePoint
  bool isFork;               //!< If the first node is a Fork
  bool isJoined;             //!< If the End of the Curve is joined.
  bool mainJoin;             //!< If the End of the Curve is joined, and this is the one Curve the others join to.
  int ID;                    //!< The ID of this TrackElement
  float startingTime;        //!< The time at which this Track begins.
  float duration;            //!< The time used to cross this TrackElement (curve).
  float endTime;             //!< The time at which this Track ends.
  float jumpTime;            //!< The Time this Track has to jump to its preceding Track (only >0 if Track isJoined==true)
  float width;               //!< Th width of the Path. This tells the Player(s), how far he(they) can go to the left/right.
  int nodeCount;             //!< The count of points this TrackElement has.
  Curve* curve;              //!< The Curve of this TrackElement
  int childCount;            //!< The number of Children This TrackElement has.
  tList<TrackElement>* children;   //!< A TrackElement can have a Tree of following TrackElements.


  // runtime
  TrackElement* history;     //!< a pointer to the last TrackElement we were on. This is if you want to walk the path backwards again.

  void debug(void) const;

  // CONDITION FUNCTIONS and STUFF
  void* subject;             //!< The Subject the Condition should act upon.
  int (TrackElement::*condFunc)(const void*) const; //!< Pointer to the condition function

  int lowest(const void* nothing) const;
  int highest(const void* nothing) const;
  int random(const void* nothing) const;

  int leftRight(const void* node) const;
  int nearest(const void* node) const;
  // todo  int enemyKilled(void* entity);

 private:
  char* name;                //!< A name for the Trac.
 
};

//! the Condition to choose between the different ways of the game.
enum CONDITION {LOWEST, HIGHEST, RANDOM, LEFTRIGHT, NEAREST, ENEMYKILLED};

//! The TrackManager handles the flow of the Players through the game.
/**

   <b>The TrackManager works as followed:</b> \n
     \n
   <b>1. Initialize it, by setting up the Graph. You can do this by using the following Commands.</b>
    \li workOn(): changes the ID that will be altered through the changes.
    \li setCurveType(): lets you set the CurveType of the Curve we are Working on. (default is BezierCurve, set this as early as possible, for this uses resources).
    \li setDuration(): sets the length of the current path in seconds.
    \li addPoint(): adds a point to the Curve.
    \li addHotPoint(): adds save/splitpoint.\n
    \li fork(): adds some interessting non-linear movments through the level (fork will force addHotPoint if not done then).
    \li condition(): decides under what condition a certain Path will be chosen.
    \li join(): joins some tracks together again. Join will set the localTime to the longest time a Path has to get to this Point)
    \li setSavePoint(): Sets a HotPoint into a savePoint. A Savepoint can be used as a rollbackpoint if a Player gets shot.

    HotPoints and Joins are at the beginning of a TrackElement. \n
    SavePoints and Forks are at the end of a TrackElement \n
    look out: <b>SAVEPOINTS CAN NOT BE FORKS</b> (but joins), because the condition is really hard to guess if you do not give some impuls. \n
\n
   <b> 2. Runtime knows the following: </b>
    \li calcPos(): returns the current position on the track
    \li calcDir(): returns the current Direction the track is flying on.
    \li tick(): makes a Step on the Path. increases localTime by dt.
    \li choosePath(): a Function that decides which Path we should follow.
   
   TrackManager can be handled as a StateMachine.
   \n\n
    Names: 
    \li TrackManager: handles Tracks
    \li Track:        The Track that the ship can follow
    \li Path:         one way through the Level, that is dependent on conditionals.
    \li Conditional:  A decition making device, that chooses betwen different TrackElements for the Path.
    \li TrackElement: A Part of A whole Track
*/
class TrackManager : public BaseObject
{
 private:
  TrackManager(void);

  static TrackManager* singletonRef;  //!< There may only be one TrackManager.

  TrackElement* firstTrackElem;       //!< The first TrackElement that exists.
  TrackElement* currentTrackElem;     //!< The TrackElement we are working on.
  CurveType curveType;                //!< The CurveType the entire TrackSystem will have.
  float localTime;                    //!< The time that has been passed since the traveling the Track.
  float maxTime;                      //!< The maximal time the track has.
  int trackElemCount;                 //!< The count of TrackElements that exist.
  PNode* bindSlave;                   //!< The node that is slave to the TrackManager. This node will be moved while update the TrackManager, and must NOT move itself.
  PNode* trackNode;                   //!< The main TrackNode of this Track.
  
  void initChildren(unsigned int childCount, TrackElement* trackElem = NULL);

 public:
  virtual ~TrackManager(void);

  static TrackManager* getInstance(void);

  // Methods to change the Path (initialisation)
  void workOn(unsigned int trackID);
  void workOn(const char* trackName);

  /** \see setCurveType(CurveType curveType, TrackElement* trackElem); \param curveType the type of the Curve */
  inline void setCurveType(CurveType curveType) { this->setCurveType (curveType, this->currentTrackElem);};
  void setCurveType(CurveType curveType, TrackElement* trackElem);
  void setDuration(float duration, TrackElement* trackElem = NULL);
  bool addPoint(Vector newPoint, TrackElement* trackElem = NULL);
  int addHotPoint(Vector newPoint, TrackElement* trackElem = NULL);
  int setSavePoint(TrackElement* trackElem = NULL);
  void fork(unsigned int count, ...);
  void forkV(unsigned int count, int* trackIDs, TrackElement* trackElem = NULL);
  void condition(unsigned int trackID, CONDITION cond, void* subject);
  void condition(CONDITION cond, void* subject, TrackElement* trackElem = NULL);
  void join(unsigned int count, ...);
  void joinc(unsigned int count, ...);
  void joinV(unsigned int count, int* trackIDs);
  void finalize(void);

  // Methods to calculate the position on the Path (runtime)
  inline Vector calcPos(void) const;
  inline Vector calcDir(void) const;
  float getWidth(void) const;
  void tick(float dt);
  void jumpTo(float time);
  inline int choosePath(TrackElement* trackElem);

  void setBindSlave(PNode* bindSlave);
  PNode* getTrackNode(void);

  // DEBUG //
  void drawGraph(float dt) const;
  void debug(unsigned int level) const;
};

#endif /* _TRACK_MANAGER_H */
