/*! 
    \file vector.h
    \brief A basic 3D math framework
    
    Contains classes to handle vectors, lines, rotations and planes
*/ 

#ifndef _VECTOR_H
#define _VECTOR_H

#include <math.h>
#include "stdincl.h"
//! PI the circle-constant
#define PI 3.14159265359f

//! 3D Vector
/**
	Class to handle 3D Vectors
*/
class Vector {

  public:
 
  float x; //!< The x Coordinate of the Vector.
  float y; //!< The y Coordinate of the Vector.
  float z; //!< The z Coordinate of the Vector.

  Vector (float x, float y, float z) : x(x), y(y), z(z) {}  //!< assignment constructor
  Vector () : x(0), y(0), z(0) {}
  ~Vector () {}

  inline Vector operator+ (const Vector& v) const { return Vector(x + v.x, y + v.y, z + v.z); }
  inline Vector operator- (const Vector& v) const { return Vector(x - v.x, y - v.y, z - v.z); }
  inline float operator* (const Vector& v) const { return x * v.x + y * v.y + z * v.z; }
  inline Vector operator* (float f) const { return Vector(x * f, y * f, z * f); }
  Vector operator/ (float f) const;
  float dot (const Vector& v) const;
  inline Vector cross (const Vector& v) const {  return Vector(y * v.z - z * v.y, z * v.x - x * v.z, x * v.y - y * v.x ); }
  void scale(const Vector& v);
  inline float len() const { return sqrt (x*x+y*y+z*z); }
  inline void normalize() { 
                      float l = len(); 
		      __UNLIKELY_IF( l == 0.0) 
			{ 
			  // Prevent divide by zero
			  return;
			}
		      x = x / l;
		      y = y / l;
		      z = z / l; 
                    }
  Vector* getNormalized();
  Vector abs();

  void debug();
};

float angleDeg (const Vector& v1, const Vector& v2);
float angleRad (const Vector& v1, const Vector& v2);

//! Quaternion
/**
	Class to handle 3-dimensional rotation efficiently
*/
class Quaternion
{
 public:
  Vector v;	//!< Imaginary Vector
  float w;        //!< Real part of the number
  
  Quaternion ();
  Quaternion (float m[4][4]);
  Quaternion (float angle, const Vector& axis);
  Quaternion (const Vector& dir, const Vector& up);
  Quaternion (float roll, float pitch, float yaw);
  
  Quaternion operator/ (const float& f) const;
  Quaternion operator* (const float& f) const;
  Quaternion operator* (const Quaternion& q) const;
  Quaternion operator+ (const Quaternion& q) const;
  Quaternion operator- (const Quaternion& q) const;
  Quaternion conjugate () const;
  Quaternion inverse () const;
  Vector apply (Vector& f) const;
  float norm () const;
  void matrix (float m[4][4]) const;
  void quatSlerp(const Quaternion* from, const Quaternion* to, const float t, Quaternion* res);
  
  void debug();
 private:
  float DELTA;      //!< resolution of calculation

};

//! 3D rotation (OBSOLETE)
/**
  Class to handle 3-dimensional rotations.
  Can create a rotation from several inputs, currently stores rotation using a 3x3 Matrix
*/
class Rotation {
  public:
  
  float m[9]; //!< 3x3 Rotation Matrix
  
  Rotation ( const Vector& v);
  Rotation ( const Vector& axis, float angle);
  Rotation ( float pitch, float yaw, float roll);
  Rotation ();
  ~Rotation () {}
  
  Rotation operator* (const Rotation& r);
  
  void glmatrix (float* buffer);
};

//!< Apply a rotation to a vector
Vector rotateVector( const Vector& v, const Rotation& r);

//! 3D line
/**
  Class to store Lines in 3-dimensional space

  Supports line-to-line distance measurements and rotation
*/
class Line
{
  public:
  
  Vector r;   //!< Offset
  Vector a;   //!< Direction
  
  Line ( Vector r, Vector a) : r(r), a(a) {}  //!< assignment constructor
  Line () : r(Vector(0,0,0)), a(Vector (1,1,1)) {}
  ~Line () {}
  
  float distance (const Line& l) const;
  float distancePoint (const Vector& v) const;
  Vector* footpoints (const Line& l) const;
  float len () const;
  
  void rotate(const Rotation& rot);
};

//! 3D plane
/**
  Class to handle planes in 3-dimensional space
  
  Critical for polygon-based collision detection
*/
class Plane
{
  public:
  
  Vector n;   //!< Normal vector
  float k;    //!< Offset constant
  
  Plane (Vector a, Vector b, Vector c);
  Plane (Vector norm, Vector p);
  Plane (Vector n, float k) : n(n), k(k) {} //!< assignment constructor
  Plane () : n(Vector(1,1,1)), k(0) {}
  ~Plane () {}
  
  Vector intersectLine (const Line& l) const;
  float distancePoint (const Vector& p) const;
  float locatePoint (const Vector& p) const;
};

#endif /* _VECTOR_H */
