/* 
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_ANIM

#include "animation_player.h"

using namespace std;


/**
   \brief standard constructor
*/
AnimationPlayer::AnimationPlayer () 
{
   this->setClassName ("AnimationPlayer");

   this->animationList = new tList<Anim>();
}

/**
   \brief the singleton reference to this class
*/
AnimationPlayer* AnimationPlayer::singletonRef = NULL;

/**
   \returns a Pointer to this Class
*/
AnimationPlayer* AnimationPlayer::getInstance(void)
{
  if (!AnimationPlayer::singletonRef)
    AnimationPlayer::singletonRef = new AnimationPlayer();
  return AnimationPlayer::singletonRef;
}

/**
   \brief standard deconstructor

   !! DANGER !! when unloading the AnimationPlayer no other Function
   should reference any Animations, from the animationList because it 
   automatically deletes them. 
   This usually happens when unloading a World.
*/
AnimationPlayer::~AnimationPlayer () 
{
  // deleting the Animation List AND all the elements of the List
  this->flush();
  delete this->animationList;

  AnimationPlayer::singletonRef = NULL;
}

/**
   \brief adds an Animation to the AnimationList.
   \param animation the Animation to handle

   when adding a Animation the Animation will too be deleted when
   the AnimationPlayer gets deleted. Consider not adding it, or 
   unadding it with animation->notHandled();
*/
void AnimationPlayer::addAnimation(Anim* animation)
{
  this->animationList->add(animation);
}

/**
   \brief removes an Animation from the Animation List, WITHOUT deleting it.
   \param animation the Anmination to remove from the List
*/
void AnimationPlayer::removeAnimation(Anim* animation)
{
  this->animationList->remove(animation);
}

/**
   \brief empties the list AND deletes all the Animations
*/ 
void AnimationPlayer::flush(void)
{
  // deleting the Animation List AND all the elements of the List
  tIterator<Anim>* animIt = this->animationList->getIterator();
  Anim* anim = animIt->nextElement();
  while( anim != NULL)
    {
      delete anim;
      this->animationList->remove(anim);
      anim = animIt->nextElement();
    }
  delete animIt;

  delete this->animationList;
  this->animationList = new tList<Anim>();
}

/**
   \brief Ticks all the animations in animationList 
   \param timePassed the time passed since the last tick.
*/
void AnimationPlayer::tick(float timePassed)
{
  tIterator<Anim>* animIt = this->animationList->getIterator();
  Anim* anim = animIt->nextElement();
  while( anim != NULL)
    {
      anim->tick(timePassed);
      anim = animIt->nextElement();
    }
  delete animIt;
}


/**
   \brief Outputs some nice debug-information
*/
void AnimationPlayer::debug(void)
{
  PRINT(0)("+------------------------------------+\n");
  PRINT(0)("+ ANIMATION PLAYER DEBUG INFORMATION +\n");
  PRINT(0)("+------------------------------------+\n");
  PRINT(0)("| Reference: %p\n", this);
  PRINT(0)("| CountOfAnims %d\n", this->animationList->getSize());
  PRINT(0)("-Animation Information---------------+\n");
  // Per ANIMATION DEBUG
  tIterator<Anim>* animIt = this->animationList->getIterator();
  Anim* anim = animIt->nextElement();
  while( anim != NULL)
    {
      //      anim->debug();
      anim = animIt->nextElement();
    }
  delete animIt;

  PRINT(0)("+--------------------------------AP--+\n");
}
