/*! 
    \file p_node.h
    \brief Definition of a parenting node

    parenting is how coordinates are handled in orxonox, meaning, that all coordinates
    are representet relative to another parent node. this nodes build a parenting
    tree of one-sided references (from up to down referenced).
    Every node manages itself a list of childrens (of whos it is parent - easy...)

    absCoordinate, absDirection have to be recalculated as soon as there was a change in
    place or ortientation. this is only the case if 
    o bDirChanged is true (so changed) AND timeStamp != now
    o bCoorChanged is true (so moved) AND timeStamp != now
    this conditions make it cheaper to recalculate the tree (reduces redundant work).

    remember: if you have to change the coordinates or the directions, use the functions
    that are defined to execute this operation - otherwhise there will be big problems...
*/


#ifndef _P_NODE_H
#define _P_NODE_H

#include "base_object.h"
#include "vector.h"

// FORWARD DEFINITION \\
class PNode; /* forward decleration, so that parentEntry has access to PNode */
//class Quaternion;
//class Vector;
template<class T> class tList;

//! enumeration for the different translation-binding-types
//typedef enum parentingMode {PNODE_LOCAL_ROTATE, PNODE_ROTATE_MOVEMENT, PNODE_ALL, PNODE_MOVEMENT, PNODE_ROTATE_AND_MOVE};
// linkage modes
#define PNODE_LOCAL_ROTATE       1    //!< Rotates all the children around their centers.
#define PNODE_ROTATE_MOVEMENT    2    //!< Moves all the children around the center of their parent, without the rotation around their own centers.
#define PNODE_MOVEMENT           4    //!< Moves all children along with the parent.
// special linkage modes
#define PNODE_ALL                3    //!< Moves all children around the center of their parent, and also rotates their centers
#define PNODE_ROTATE_AND_MOVE    5    //!< Rotates all children around their axis, and moves them as the Parent Moves, but does not rotate around the center of their parent.

//! The default mode of the translation-binding.
#define DEFAULT_MODE PNODE_ALL

//! Patent Node is a Engine to calculate the position of an Object in respect to the position of its parent.
class PNode : public BaseObject {

 public:
  PNode ();
  PNode (const Vector& absCoordinate, PNode* pNode);
  virtual ~PNode ();

  PNode* parent;            //!< a pointer to the parent node
  tList<PNode>* children;   //!< list of the children



  inline Vector* getRelCoor () const { return this->relCoordinate; }
  void setRelCoor (const Vector& relCoord);
  inline Vector getAbsCoor () const { return *this->absCoordinate; }
  void setAbsCoor (const Vector& absCoord);
  void shiftCoor (Vector* shift);
  void shiftCoor (const Vector& shift);

  inline Quaternion getRelDir () const { return *this->relDirection; }
  //void setRelDir (Quaternion* relDir);
  void setRelDir (const Quaternion& relDir);
  inline Quaternion getAbsDir () const { return *this->absDirection; }
  //void setAbsDir (Quaternion* absDir);
  void setAbsDir (const Quaternion& absDir);
  //void shiftDir (Quaternion* shift);
  void shiftDir (const Quaternion& shift);

  float getSpeed() const;

  void addChild (PNode* pNode, int parentingMode = DEFAULT_MODE);
  void removeChild (PNode* pNode);
  void remove();


  void setParent (PNode* parent);
  void parentCoorChanged ();
  void parentDirChanged ();
  void setMode (int parentingMode);
  int getMode();

  virtual void update (float dt);
  void processTick (float dt);

  void setName (char* newName);
  char* getName ();


  void debug ();

 protected:
  float timeStamp;         //!< this the timeStamp of when the abs{Coordinat, Direction} has been calculated
  char* objectName;        //!< The name of the Object
  bool bAbsCoorChanged;    //!< If Absolute Coordinate has changed since last time we checked
  bool bRelCoorChanged;    //!< If Relative Coordinate has changed since last time we checked
  bool bAbsDirChanged;     //!< If Absolute Direction has changed since last time we checked
  bool bRelDirChanged;     //!< If Relative Direction has changed since last time we checked

  Vector* relCoordinate;    //!< coordinates relative to the parent
  Vector* absCoordinate;    //!< absolute coordinates in the world ( from (0,0,0) )
  Quaternion* relDirection; //!< direction relative to the parent
  Quaternion* absDirection; //!< absolute direvtion in the world ( from (0,0,1) )

  int mode;                //!< the mode of the binding

 private:
  void init(PNode* parent);

  Vector* lastAbsCoordinate; //!< this is used for speedcalculation, it stores the last coordinate
  Vector* diffCoordinate;    //!< this is stored here for performance reasons, difference to the last vector
  float time;                //!< time since last update
};

#endif /* _P_NODE_H */
 
